/***********************************************************
*                        DCF-clock                         *
*            a tiny DCF controlled digital clock           *
*                                                          *
*           by Frank Goetze  -  www.embedded-os.de         *
************************************************************
* Permission to use, copy, modify, and distribute this     *
* software in source and binary forms and its              *
* documentation for any purpose and without fee is hereby  *
* granted, provided that the above authors notice appear   *
* in all copies and that both that authors notice and this *
* permission notice appear in supporting documentation.    *
*                                                          *
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND   *
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT    *
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY    *
* AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.     *
* IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR             *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,    *
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF     *
* USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER *
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN        *
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING           *
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE   *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY *
* OF SUCH DAMAGE.                                          *
************************************************************
*               clock_Adafruit_Feather_M0.c                *
*   the board support package for the Adafruit Feather M0  *
************************************************************
* Revision-Info:                                           *
*                                                          *
* 1.00a     first alpa-release                             *
*                                                          *
***********************************************************/
#include <samd21.h>

#include "../BSP/clock_Adafruit_Feather_M0.h"

extern void SysTick_Handler_time(void);

/******************************************************************************
 *   Atmel SAMD21G18 configuration
 ******************************************************************************
 *
 * we use the external 32.768 khz crystal oscillator and GCLK Generator 1 and
 * feed this clock to the DFLL48M FLL and provide this to the CPU
 *
 *                                                  user LED
 *   PA17 ------------------------------------- >I ---|
 *
 *                                                  user Button
 *   PA15 -------------------------------------- \ ---|     (internal pull-up)
 *
 *****
 *                                                  UART to DCF-receiver (HKW-shop)
 *   PA10 --- SERCOM2.2 ----------------------- TXD0    ->    (unused)
 *   PA11 --- SERCOM2.3 ----------------------- RXD0    <-    (NO internal pull-up possible !)
 *
 *   : OR :
 *                                                  pure DCF77 receiver
 *   PA10 ------------------------------------- DCF_en  ->    low active (unused)
 *   PA11 ------------------------------------- DCF     <-    (internal pull-up)
 *
 *****
 *                                                  SPI to OLED display
 *   PB10 --- SERCOM4.2 ----------------------- MOSI    ->
 *   PB11 --- SERCOM4.3 ----------------------- SCK     ->
 *   PB08 ------------------------------------- /CS     ->    (internal pull-up)
 *   PB09 ------------------------------------- D/C     ->    (internal pull-up)
 *   PB02 ------------------------------------- /RES    ->    (internal pull-up)
 *
 *   : OR :
 *                                                  I2C to OLED display
 *   PA22 --- SERCOM5.0 ----------------------- SDA     ->    (NO internal pull-up possible !)
 *   PA23 --- SERCOM5.1 ----------------------- SCL     ->    (NO internal pull-up possible !)
 *
 *****
 *                                                  ADC
 *   PA03 ------------------------------------- VREFA   <- 3.3V
 *   PA07 ------------------------------------- Vbat/2  <-
 *
 *
 ******************************************************************************/

enum {
    PORTA,
    PORTB,
    PORTC
};
//-------------------------------- Pin name on Adafruit Feather M0 Basic Proto
#define LED_GROUP     PORTA
#define LED_PORT      PORT_PA17               // D13
#define LED_PIN       PIN_PA17

#define BUTTON_GROUP  PORTA
#define BUTTON_PORT   PORT_PA15               // D5
#define BUTTON_PIN    PIN_PA15 

#ifdef DCF77
 #define DCF_GROUP    PORTA
 #define DCF_en_PORT  PORT_PA10               // D1
 #define DCF_in_PORT  PORT_PA11               // D0
 #define DCF_en_PIN   PIN_PA10 
 #define DCF_in_PIN   PIN_PA11 
#else
 #define UART_GROUP   PORTA
 #define UART_PORT_TX PORT_PA10               // D1
 #define UART_PORT_RX PORT_PA11               // D0
 #define UART_PIN_TX  PIN_PA10
 #define UART_PIN_RX  PIN_PA11
#endif

#define I2C_GROUP     PORTA
#define I2C_PORT_SDA  PORT_PA22               // SDA
#define I2C_PORT_SCL  PORT_PA23               // SCL
#define I2C_PIN_SDA   PIN_PA22
#define I2C_PIN_SCL   PIN_PA23

#define SPI_GROUP     PORTB
#define SPI_PORT_MOSI 1 << 10  // PORT_PB10   // MOSI
#define SPI_PORT_SCK  1 << 11  // PORT_PB11   // SCK
#define SPI_PIN_MOSI  10       // PIN_PB10
#define SPI_PIN_SCK   11       // PIN_PB11
#define SPI_PORT_CS   PORT_PB08               // A1
#define SPI_PORT_DC   PORT_PB09               // A2
#define SPI_PORT_RES  PORT_PB02               // A5
#define SPI_PIN_CS    PIN_PB08
#define SPI_PIN_DC    PIN_PB09
#define SPI_PIN_RES   PIN_PB02

#define ADC_GROUP     PORTA
#define ADC_PORT_AD   PORT_PA07               // D9
#define VBAT_PIN      PIN_PA07
#define VREF_PIN      PIN_PA03                // AREF

//#define SERCOM_SPI_BAUDRATE(x)   ((uint64_t)65536 * (SystemCoreClock - (16 * x)) / SystemCoreClock)
//#define SERCOM_I2C_BAUDRATE(x)   ((uint64_t)65536 * (SystemCoreClock - (16 * x)) / SystemCoreClock)

unsigned int Button_Get(void);
/*
************************************************************
*                   sys-tick & delay                       *
************************************************************
*/
volatile unsigned int msTicks;    /* counts 10ms timeTicks */
/*
 * corretion for too fast running clock crystals
 *
 *                     ticks per 24h                   8640000
 * coor_cnt =  --------------------------------- = ---------------- = 1440 (1 min per 24h)
 *               ticks of time too fast in 24h            6000
 *
 * incorrect tick based on the 32.768KHz crystal mutiplied to 
 * 48.005120 MHz and with 100 ticks per secound is:   
 *             0.2 * 60 * 60 * 24 = 17280
 *
 */
#define corr_cnt 17280
#if (corr_cnt > 0)
 volatile unsigned int corr_val = 0; 
#endif
void SysTick_Handler(void)
{
    msTicks++;
#if (corr_cnt > 0)
    corr_val++;
    if (corr_val < corr_cnt) SysTick_Handler_time();
    else corr_val = 0;
#else
    SysTick_Handler_time();
#endif
}

void BSP_delay(unsigned int dlyTicks)
{
    unsigned int curTicks;

    curTicks = msTicks;
    while (((msTicks - curTicks) < dlyTicks) && (!Button_Get())) { __NOP(); }
}

/*
************************************************************
*                    48MHz sys-clock                       *
************************************************************
*/
void SetupMainClock(void)
{
    SYSCTRL->XOSC32K.reg  = SYSCTRL_XOSC32K_ENABLE;
    SYSCTRL->XOSC32K.reg |= SYSCTRL_XOSC32K_STARTUP(5) | SYSCTRL_XOSC32K_AAMPEN | SYSCTRL_XOSC32K_EN32K | 
                            SYSCTRL_XOSC32K_XTALEN | SYSCTRL_XOSC32K_ENABLE;
    while((SYSCTRL->PCLKSR.reg & SYSCTRL_PCLKSR_XOSC32KRDY) == 0) __NOP();
    GCLK->GENDIV.reg  = GCLK_GENDIV_DIV(1) | GCLK_GENDIV_ID(1);
    GCLK->GENCTRL.reg = GCLK_GENCTRL_ID(1) | GCLK_GENCTRL_SRC_XOSC32K | GCLK_GENCTRL_GENEN;
    GCLK->CLKCTRL.reg = GCLK_CLKCTRL_GEN_GCLK1 | GCLK_CLKCTRL_CLKEN | GCLK_CLKCTRL_ID(0) ;
    SYSCTRL->DFLLCTRL.reg  = SYSCTRL_DFLLCTRL_ENABLE;
    SYSCTRL->DFLLMUL.reg   = SYSCTRL_DFLLMUL_CSTEP(5) | SYSCTRL_DFLLMUL_FSTEP(10); 	
    SYSCTRL->DFLLMUL.reg  |= SYSCTRL_DFLLMUL_MUL(1465); // 1465 * 32768 is approx 48MHZ (48.005120 MHz)
    SYSCTRL->DFLLCTRL.reg |= SYSCTRL_DFLLCTRL_MODE;
    while((SYSCTRL->PCLKSR.reg & SYSCTRL_PCLKSR_DFLLLCKC) == 0) __NOP();
    NVMCTRL->CTRLB.reg |= NVMCTRL_CTRLB_RWS(1);
    GCLK->GENDIV.reg  = GCLK_GENDIV_DIV(1) | GCLK_GENDIV_ID(0);
    GCLK->GENCTRL.reg = GCLK_GENCTRL_GENEN | GCLK_GENCTRL_SRC_DFLL48M | GCLK_GENCTRL_ID(0);
    GCLK->CLKCTRL.reg = GCLK_CLKCTRL_CLKEN | GCLK_CLKCTRL_GEN_GCLK0 | GCLK_CLKCTRL_ID_DFLL48;
}

/*
************************************************************
*                          ADC                             *
************************************************************
*/
#define ADC_GCLK3_PreDiv 6
void ADC_Init(void)
{
    PM->APBCMASK.reg  |= PM_APBCMASK_ADC;
    GCLK->GENDIV.reg   = GCLK_GENDIV_DIV(ADC_GCLK3_PreDiv) | GCLK_GENDIV_ID(3);
    GCLK->GENCTRL.reg  = GCLK_GENCTRL_GENEN | GCLK_GENCTRL_SRC_DFLL48M | GCLK_GENCTRL_ID(3);
    GCLK->CLKCTRL.reg  = GCLK_CLKCTRL_CLKEN | GCLK_CLKCTRL_GEN_GCLK3 | GCLK_CLKCTRL_ID_ADC;
    ADC->CTRLA.reg = ADC_CTRLA_SWRST;
    while(ADC->CTRLA.bit.SWRST) __NOP();
    ADC->REFCTRL.reg   = ADC_REFCTRL_REFCOMP | ADC_REFCTRL_REFSEL_AREFA;
    ADC->AVGCTRL.reg   = ADC_AVGCTRL_ADJRES(0) | ADC_AVGCTRL_SAMPLENUM_1;
    ADC->SAMPCTRL.reg  = ADC_SAMPCTRL_SAMPLEN(0);
    while(ADC->STATUS.bit.SYNCBUSY) __NOP();
    ADC->CTRLB.reg     = ADC_CTRLB_PRESCALER_DIV512 | ADC_CTRLB_RESSEL_12BIT;
    while(ADC->STATUS.bit.SYNCBUSY) __NOP();
    ADC->INPUTCTRL.reg = ADC_INPUTCTRL_GAIN_1X | ADC_INPUTCTRL_INPUTOFFSET(0) | ADC_INPUTCTRL_INPUTSCAN(0) | 
                         ADC_INPUTCTRL_MUXNEG_GND | ADC_INPUTCTRL_MUXPOS_PIN7;
    while(ADC->STATUS.bit.SYNCBUSY) __NOP();
    ADC->CALIB.reg     = ADC_CALIB_BIAS_CAL(*(uint32_t*)ADC_FUSES_BIASCAL_ADDR >> ADC_FUSES_BIASCAL_Pos) | 
                         ADC_CALIB_LINEARITY_CAL(*(uint64_t*)ADC_FUSES_LINEARITY_0_ADDR >> ADC_FUSES_LINEARITY_0_Pos);
    while(ADC->STATUS.bit.SYNCBUSY) __NOP();
    PORT->Group[ADC_GROUP].PINCFG[VREF_PIN].bit.PMUXEN = 1;
    PORT->Group[ADC_GROUP].PINCFG[VBAT_PIN].bit.PMUXEN = 1;
    PORT->Group[ADC_GROUP].PMUX[VREF_PIN/2].reg |= PORT_PMUX_PMUXO_B;
    PORT->Group[ADC_GROUP].PMUX[VBAT_PIN/2].reg |= PORT_PMUX_PMUXO_B;
    ADC->CTRLA.reg |= ADC_CTRLA_ENABLE;
    while(ADC->STATUS.bit.SYNCBUSY) __NOP();
}

float ADC_VBATread(void)
{
    float measuredvbat;
    
    ADC->SWTRIG.reg = ADC_SWTRIG_START | ADC_SWTRIG_FLUSH;
    while (ADC->STATUS.reg & ADC_STATUS_SYNCBUSY);
    while (ADC->INTFLAG.bit.RESRDY == 0);
    
    measuredvbat = (float)ADC->RESULT.reg;
    
//    measuredvbat /= 4;     // divide by the 4 samples
    measuredvbat *= 2;     // we divided by 2 with resistors, so multiply back
    measuredvbat *= 3.3;   // Multiply by 3.3, based on our VREFA 3.3V
    measuredvbat /= 4095;  // convert 2^12 resolution to voltage
    return measuredvbat;
}    

/*
************************************************************
*                          I2C                             *
************************************************************
*/
#define I2C_Baudrate   1000000
#define I2C_slave_Addr    0x3C
void I2C_Init(void)
{
    PM->APBCMASK.reg  |= PM_APBCMASK_SERCOM5;
    GCLK->CLKCTRL.reg = GCLK_CLKCTRL_CLKEN | GCLK_CLKCTRL_GEN_GCLK0 | GCLK_CLKCTRL_ID_SERCOM5_CORE;
    SERCOM5->I2CM.CTRLA.reg = SERCOM_I2CM_CTRLA_SWRST;
    while(SERCOM5->I2CM.SYNCBUSY.bit.SWRST) __NOP();
    SERCOM5->I2CM.CTRLA.reg = SERCOM_I2CM_CTRLA_SPEED(1) | SERCOM_I2CM_CTRLA_SDAHOLD(0x2) | 
                              SERCOM_I2CM_CTRLA_RUNSTDBY | SERCOM_I2CM_CTRLA_MODE_I2C_MASTER;
    SERCOM5->I2CM.CTRLB.reg = SERCOM_I2CM_CTRLB_SMEN;
    while(SERCOM5->I2CM.SYNCBUSY.bit.SYSOP) __NOP();
    SERCOM5->I2CM.BAUD.reg = SERCOM_I2CM_BAUD_BAUD(11) | SERCOM_I2CM_BAUD_BAUDLOW(22); //SERCOM_I2C_BAUDRATE(I2C_Baudrate);
    while(SERCOM5->I2CM.SYNCBUSY.bit.SYSOP) __NOP();
    PORT->Group[I2C_GROUP].PINCFG[I2C_PIN_SDA].bit.PMUXEN = 1;
    PORT->Group[I2C_GROUP].PINCFG[I2C_PIN_SCL].bit.PMUXEN  = 1;
    PORT->Group[I2C_GROUP].PMUX[I2C_PIN_SDA/2].reg = (PORT_PMUX_PMUXE_D | PORT_PMUX_PMUXO_D);
    SERCOM5->I2CM.CTRLA.reg |= SERCOM_I2CM_CTRLA_ENABLE;
    while(SERCOM5->I2CM.SYNCBUSY.bit.ENABLE) __NOP();
    while(!(SERCOM5->I2CM.STATUS.reg & SERCOM_I2CM_STATUS_BUSSTATE(1))) {
        SERCOM5->I2CM.STATUS.reg = SERCOM_I2CM_STATUS_BUSSTATE(1); 
    }
}

void I2C_send(uint8_t *data, uint16_t len)
{
    #define COMMAND_BYTE        0x00
    #define DATA_BYTES          0x40
    uint16_t   i = 0;
    uint8_t type = COMMAND_BYTE;
    
    while(SERCOM5->I2CM.SYNCBUSY.reg & SERCOM_I2CM_SYNCBUSY_MASK) __NOP();
    SERCOM5->I2CM.CTRLB.reg &= ~SERCOM_I2CM_CTRLB_ACKACT;
    while(SERCOM5->I2CM.SYNCBUSY.bit.SYSOP) __NOP();
    SERCOM5->I2CM.ADDR.reg = (I2C_slave_Addr << 1) | (0 << SERCOM_I2CM_ADDR_HS_Pos);
    while(!(SERCOM5->I2CM.INTFLAG.reg & SERCOM_I2CM_INTFLAG_MB) && !(SERCOM5->I2CM.INTFLAG.reg & SERCOM_I2CM_INTFLAG_SB))
    if(SERCOM5->I2CM.INTFLAG.reg & SERCOM_I2CM_INTFLAG_SB) {
        SERCOM5->I2CM.INTFLAG.reg = SERCOM_I2CM_INTFLAG_SB;
        if(SERCOM5->I2CM.STATUS.reg & SERCOM_I2CM_STATUS_ARBLOST) return;
        else if(SERCOM5->I2CM.STATUS.reg & SERCOM_I2CM_STATUS_RXNACK) goto I2C_err;
    }

    if(len > 1) type = DATA_BYTES;
    if (!(SERCOM5->I2CM.STATUS.reg & SERCOM_I2CM_STATUS_BUSSTATE(2))) goto I2C_err;
    while(SERCOM5->I2CM.SYNCBUSY.reg & SERCOM_I2CM_SYNCBUSY_MASK) __NOP();
    SERCOM5->I2CM.DATA.reg = type;
    while (!(SERCOM5->I2CM.INTFLAG.reg & SERCOM_I2CM_INTFLAG_MB) && !(SERCOM5->I2CM.INTFLAG.reg & SERCOM_I2CM_INTFLAG_SB)) __NOP();
    if (SERCOM5->I2CM.STATUS.reg & SERCOM_I2CM_STATUS_RXNACK) goto I2C_err;

    while(i < len) {
        if (!(SERCOM5->I2CM.STATUS.reg & SERCOM_I2CM_STATUS_BUSSTATE(2))) goto I2C_err;
        while(SERCOM5->I2CM.SYNCBUSY.reg & SERCOM_I2CM_SYNCBUSY_MASK) __NOP();
        SERCOM5->I2CM.DATA.reg = data[i++];
        while (!(SERCOM5->I2CM.INTFLAG.reg & SERCOM_I2CM_INTFLAG_MB) && !(SERCOM5->I2CM.INTFLAG.reg & SERCOM_I2CM_INTFLAG_SB)) __NOP();
        if (SERCOM5->I2CM.STATUS.reg & SERCOM_I2CM_STATUS_RXNACK) goto I2C_err;
    }
I2C_err:
    while(SERCOM5->I2CM.SYNCBUSY.reg & SERCOM_I2CM_SYNCBUSY_MASK) __NOP();
    SERCOM5->I2CM.CTRLB.reg |= SERCOM_I2CM_CTRLB_CMD(3);
}

/*
************************************************************
*                          SPI                             *
************************************************************
*/
#define SPI_Baudrate 48000000
void SPI_Init(void)
{
    PM->APBCMASK.reg  |= PM_APBCMASK_SERCOM4;
    GCLK->CLKCTRL.reg = GCLK_CLKCTRL_CLKEN | GCLK_CLKCTRL_GEN_GCLK0 | GCLK_CLKCTRL_ID_SERCOM4_CORE;
    SERCOM4->SPI.CTRLA.reg = SERCOM_SPI_CTRLA_SWRST;
    while(SERCOM4->SPI.SYNCBUSY.bit.SWRST) __NOP();
    SERCOM4->SPI.CTRLA.reg = SERCOM_SPI_CTRLA_DOPO(1) | SERCOM_SPI_CTRLA_MODE_SPI_MASTER;
    SERCOM4->SPI.BAUD.reg  = 1; //SERCOM_SPI_BAUDRATE(SPI_Baudrate);
    SERCOM4->SPI.CTRLB.reg = 0;
    while(SERCOM4->SPI.SYNCBUSY.bit.CTRLB) __NOP();
    PORT->Group[SPI_GROUP].PINCFG[SPI_PIN_MOSI].bit.PMUXEN = 1;
    PORT->Group[SPI_GROUP].PINCFG[SPI_PIN_SCK].bit.PMUXEN  = 1;
    PORT->Group[SPI_GROUP].PMUX[SPI_PIN_MOSI/2].reg = (PORT_PMUX_PMUXE_D | PORT_PMUX_PMUXO_D);
    SERCOM4->SPI.CTRLA.reg |= SERCOM_SPI_CTRLA_ENABLE;
    while(SERCOM4->SPI.SYNCBUSY.bit.ENABLE) __NOP();
    // config the additional manual lines /CS, D/C and /RES
    PORT->Group[SPI_GROUP].DIRSET.reg = SPI_PORT_CS;
    PORT->Group[SPI_GROUP].OUTSET.reg = SPI_PORT_CS;
    PORT->Group[SPI_GROUP].DIRSET.reg = SPI_PORT_DC;
    PORT->Group[SPI_GROUP].OUTSET.reg = SPI_PORT_DC;
    PORT->Group[SPI_GROUP].DIRSET.reg = SPI_PORT_RES;
    PORT->Group[SPI_GROUP].OUTSET.reg = SPI_PORT_RES;
}

void SPI_write(uint8_t data)
{
    while(!(SERCOM4->SPI.INTFLAG.reg & SERCOM_SPI_INTFLAG_DRE)) __NOP();
    SERCOM4->SPI.DATA.reg = data;
    while(!(SERCOM4->SPI.INTFLAG.reg & SERCOM_SPI_INTFLAG_TXC)) __NOP();
}

void SPI_CS_en(void)   { PORT->Group[SPI_GROUP].OUTCLR.reg = SPI_PORT_CS;  }
void SPI_CS_dis(void)  { PORT->Group[SPI_GROUP].OUTSET.reg = SPI_PORT_CS;  }

void SPI_DC_en(void)   { PORT->Group[SPI_GROUP].OUTCLR.reg = SPI_PORT_DC;  }
void SPI_DC_dis(void)  { PORT->Group[SPI_GROUP].OUTSET.reg = SPI_PORT_DC;  }

void SPI_RES_en(void)  { PORT->Group[SPI_GROUP].OUTCLR.reg = SPI_PORT_RES; }
void SPI_RES_dis(void) { PORT->Group[SPI_GROUP].OUTSET.reg = SPI_PORT_RES; }


#ifdef DCF77
/*
************************************************************
*                         DCF77                            *
************************************************************
*/
void DCF_Init(void)
{
    PORT->Group[DCF_GROUP].DIRSET.reg = DCF_en_PORT;
    PORT->Group[DCF_GROUP].OUTCLR.reg = DCF_en_PORT;

    PORT->Group[DCF_GROUP].OUT.reg           |=  DCF_in_PORT;
    PORT->Group[DCF_GROUP].PINCFG[DCF_in_PIN].bit.INEN   = 1;
    PORT->Group[DCF_GROUP].PINCFG[DCF_in_PIN].bit.PULLEN = 1;
    PORT->Group[DCF_GROUP].DIRCLR.reg         =  DCF_in_PORT;
}

unsigned int DCF_in_Get(void) { return !((PORT->Group[DCF_GROUP].IN.reg) & (DCF_in_PORT)); }

void DCF_Enable(void)  { PORT->Group[DCF_GROUP].OUTCLR.reg = DCF_en_PORT; }
void DCF_Disable(void) { PORT->Group[DCF_GROUP].OUTSET.reg = DCF_en_PORT; }

#else

/*
************************************************************
*                         UART                             *
************************************************************
*/
uint8_t rx_data[18];
uint8_t rx_i = 0;
#define UART_Baudrate 300
#define UART_GCLK2_PreDiv 16
void UART_Init(void)
{
    PM->APBCMASK.reg |= PM_APBCMASK_SERCOM2;
    GCLK->GENDIV.reg  = GCLK_GENDIV_DIV(UART_GCLK2_PreDiv) | GCLK_GENDIV_ID(2);
    GCLK->GENCTRL.reg = GCLK_GENCTRL_GENEN | GCLK_GENCTRL_SRC_DFLL48M | GCLK_GENCTRL_ID(2);
    GCLK->CLKCTRL.reg = GCLK_CLKCTRL_CLKEN | GCLK_CLKCTRL_GEN_GCLK2 | GCLK_CLKCTRL_ID_SERCOM2_CORE;
    SERCOM2->USART.CTRLA.reg = SERCOM_USART_CTRLA_SWRST;
    while(SERCOM2->USART.SYNCBUSY.bit.SWRST) __NOP();
    SERCOM2->USART.CTRLA.reg = SERCOM_USART_CTRLA_DORD | SERCOM_USART_CTRLA_FORM(1) | 
                               SERCOM_USART_CTRLA_RXPO(3) | SERCOM_USART_CTRLA_TXPO(1) | 
                               SERCOM_USART_CTRLA_MODE_USART_INT_CLK;
    SERCOM2->USART.BAUD.reg  = SERCOM_BAUDRATE(UART_Baudrate * UART_GCLK2_PreDiv);
    SERCOM2->USART.CTRLB.reg = SERCOM_USART_CTRLB_RXEN | SERCOM_USART_CTRLB_TXEN | 
                               /*SERCOM_USART_CTRLB_PMODE | */ SERCOM_USART_CTRLB_SBMODE |
                               SERCOM_USART_CTRLB_CHSIZE(7);
    while(SERCOM2->USART.SYNCBUSY.bit.CTRLB) __NOP();
    PORT->Group[UART_GROUP].PINCFG[UART_PIN_TX].bit.PMUXEN = 1;
    PORT->Group[UART_GROUP].PINCFG[UART_PIN_RX].bit.PMUXEN = 1;
    PORT->Group[UART_GROUP].PMUX[UART_PIN_TX/2].reg = (PORT_PMUX_PMUXE_D | PORT_PMUX_PMUXO_D);
    SERCOM2->USART.INTENSET.reg = SERCOM_USART_INTENSET_RXC;
    NVIC_EnableIRQ(SERCOM2_IRQn);
    SERCOM2->USART.CTRLA.reg |= SERCOM_USART_CTRLA_ENABLE;
    while(SERCOM2->USART.SYNCBUSY.bit.ENABLE) __NOP();
}

void write_uart_data(uint8_t data)
{
    while((SERCOM2->USART.INTFLAG.bit.DRE) == 0) __NOP();
    SERCOM2->USART.DATA.reg = data;
}
uint8_t read_uart_data(void)
{
    return SERCOM2->USART.DATA.reg;
}

void SERCOM2_Handler(void)
{
    uint8_t d;
    
    if (SERCOM2->USART.INTFLAG.bit.RXC) {
        d = SERCOM2->USART.DATA.reg;
        if ((d == 0x13) || ((rx_i + 1) >= sizeof(rx_data))) { 
            rx_data[rx_i] = 0; 
            rx_i = 0;
        }
        else rx_data[rx_i++] = d;
        if (d == 0x13) rx_det = 2; 
        else rx_det = 1;
    }
    if (SERCOM2->USART.INTFLAG.bit.TXC) {
        SERCOM2->USART.INTFLAG.reg = SERCOM_USART_INTFLAG_TXC;
        // write next    
    }
}

#endif


/*
************************************************************
*                      user-button                         *
************************************************************
*/
void Button_Init(void)
{
    PORT->Group[BUTTON_GROUP].OUT.reg           |=  BUTTON_PORT;
    PORT->Group[BUTTON_GROUP].PINCFG[BUTTON_PIN].bit.INEN   = 1;
    PORT->Group[BUTTON_GROUP].PINCFG[BUTTON_PIN].bit.PULLEN = 1;
    PORT->Group[BUTTON_GROUP].DIRCLR.reg         =  BUTTON_PORT;
}

unsigned int Button_Get(void)
{
    return (~(PORT->Group[BUTTON_GROUP].IN.reg) & (BUTTON_PORT));
}


/*
************************************************************
*                       user-LED                           *
************************************************************
*/
void LED_Init(void)
{
    PORT->Group[LED_GROUP].DIRSET.reg = LED_PORT;
    PORT->Group[LED_GROUP].OUTCLR.reg = LED_PORT;
}

void LED_On(void) 
{
    PORT->Group[LED_GROUP].OUTSET.reg = LED_PORT;
}

void LED_Off(void)
{
    PORT->Group[LED_GROUP].OUTCLR.reg = LED_PORT;
}

void LED_Toggle(void)
{
    PORT->Group[LED_GROUP].OUTTGL.reg = LED_PORT;
}



void BSP_Init(void)
{
    Button_Init();
    LED_Init();
#ifdef OLED_SPI
    SPI_Init();
#elif defined OLED_I2C
    I2C_Init();
#else
 #warning "unknown display type!"
#endif
#ifdef DCF77
    DCF_Init();
#else
    UART_Init();
#endif
    ADC_Init();
}

/************************* END ****************************/
