/***********************************************************
*                        DCF-clock                         *
*            a tiny DCF controlled digital clock           *
*                                                          *
*           by Frank Goetze  -  www.embedded-os.de         *
************************************************************
* Permission to use, copy, modify, and distribute this     *
* software in source and binary forms and its              *
* documentation for any purpose and without fee is hereby  *
* granted, provided that the above authors notice appear   *
* in all copies and that both that authors notice and this *
* permission notice appear in supporting documentation.    *
*                                                          *
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND   *
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT    *
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY    *
* AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.     *
* IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR             *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,    *
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF     *
* USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER *
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN        *
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING           *
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE   *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY *
* OF SUCH DAMAGE.                                          *
************************************************************
*                      clock_pong.c                        *
*           a pong clock with the time as score            *
************************************************************
*   pong clock bases on the arduino code from Rob Parrett  *
*         at: https://github.com/rparrett/pongclock        *
************************************************************
* Revision-Info:                                           *
*                                                          *
* 1.00a     first alpa-release                             *
*                                                          *
***********************************************************/
#include <samd21.h>

#include <stdlib.h>
#include <math.h>

#include "../inc/clock_time.h"
#include "../inc/clock_display.h"
#include "../display/OLED_SSD1306.h"
#include "../BSP/clock_Adafruit_Feather_M0.h"

extern time_t  time;
extern char datetime[16];

/************************ DEFS ****************************/
typedef struct _display {
    short  h;
    short  w;
} display_t;
display_t display = {DISPLAY_Y, DISPLAY_X};

typedef struct _paddle {
    short  h;
    short  w;
} paddle_t;
paddle_t  paddle = {PADDLE_H, PADDLE_W};

typedef struct _lpaddle {
    short  x;
    short  y;
    short  d;
    short  ball;
} lpaddle_t;
lpaddle_t  lpaddle = {0, PADDLE_W, 1, DISPLAY_X - DISPLAY_X / 4};

typedef struct _rpaddle {
    short  x;
    short  y;
    short  d;
    short  ball;
} rpaddle_t;
rpaddle_t  rpaddle = {DISPLAY_X - PADDLE_W, DISPLAY_Y - PADDLE_H, -1, DISPLAY_X / 4};

typedef struct _ball {
    short  x;
    short  y;
    short  dx;
    short  dy;
    short  w;
    short  h;
} ball_t;
ball_t  ball = {PADDLE_W, PADDLE_W, 1, 1, BALL_W, BALL_H};

typedef struct _dashline {
    short  h;
    short  w;
    short  n;
    short  x;
    short  y;
} dashline_t;
dashline_t  dashline = {DASHLINE_H, DASHLINE_W, DISPLAY_Y / DASHLINE_H, DISPLAY_X / 2 - 1, DASHLINE_H / 2};

#define FONTSIZE  2
#define FONT_S  FONTSIZE
typedef struct _font {
    short  s;
    short  w;
    short  h;
    short  sp;
} font_pt;
font_pt  font = {FONT_S, 8 * FONT_S, 11 * FONT_S, 3 * FONT_S};
 
short lscore;
short rscore;

short target_y = 0;

int pause = 0;


#define fgcolor  1
#define bgcolor  0

/************************* HELP ******************************/
static long absolute(long value)
{
  if (value < 0) return -value;
  else return value;  
}

static int random(unsigned short min, unsigned short max)
{
    unsigned int rng;

    rng = rand();
    while(rng > max) rng /= 10;
    return rng;
}

static void t_itoa(char *buff, unsigned int value, unsigned int s_div, unsigned char mode)
{
    if ((value / s_div) > 0) { *buff++ = (value / s_div) + '0'; value %= s_div; }
    else if (mode != 0) *buff++ = '0';
    *buff++ = value + '0';
    *buff++ = 0;
}

/************************* FUNC ******************************/

void midline()
{
    unsigned short  i;

    for(i = 0; i < dashline.n; i = i + 4) {
        OLED_fillRect(dashline.x, dashline.y + i * dashline.h, dashline.w, dashline.h, 1);
    }
}

void score(short color)
{
    short lscore_x, rscore_x;
    char str[6];

    lscore_x = display.w / 2 - font.w - font.w - font.sp - 1;
    rscore_x = display.w / 2 + font.sp;
    t_itoa(str, lscore, 10, 1); 
    if (lscore < 10) str[0] = ' ';
    OLED_Text(lscore_x, dashline.y + 3, font_Dot, str);
    t_itoa(str, rscore, 10, 1);
    OLED_Text(rscore_x, dashline.y + 3, font_Dot, str);
}

void leftpaddle()
{
    if (pause > 0) return;
    if (lpaddle.d == 1)       OLED_fillRect(lpaddle.x, lpaddle.y, paddle.w, 1, bgcolor);
    else if (lpaddle.d == -1) OLED_fillRect(lpaddle.x, lpaddle.y + paddle.h - 1, paddle.w, 1, bgcolor);
    lpaddle.y = lpaddle.y + lpaddle.d;
    if (ball.dx == 1) lpaddle.d = 0;
    else {
        if (lpaddle.y + paddle.h / 2 == target_y)     lpaddle.d =  0;
        else if (lpaddle.y + paddle.h / 2 > target_y) lpaddle.d = -1;
        else                                          lpaddle.d =  1;
    }
    if (lpaddle.y + 2*paddle.w + paddle.h >= display.h && lpaddle.d == 1)  lpaddle.d = 0;
    else if (lpaddle.y <= 2*paddle.w && lpaddle.d == -1)                   lpaddle.d = 0;
    OLED_fillRect(lpaddle.x, lpaddle.y, paddle.w, paddle.h, fgcolor);
}

void rightpaddle()
{
    if (pause > 0) return;
    if (rpaddle.d == 1)       OLED_fillRect(rpaddle.x, rpaddle.y, paddle.w, 1, bgcolor);
    else if (rpaddle.d == -1) OLED_fillRect(rpaddle.x, rpaddle.y + paddle.h - 1, paddle.w, 1, bgcolor);
    rpaddle.y = rpaddle.y + rpaddle.d;
    if (ball.dx == -1) rpaddle.d = 0;
    else {
        if (rpaddle.y + paddle.h / 2 == target_y)     rpaddle.d =  0;
        else if (rpaddle.y + paddle.h / 2 > target_y) rpaddle.d = -1;
        else                                          rpaddle.d =  1;
    }
    if (rpaddle.y + 2*paddle.w + paddle.h >= display.h && rpaddle.d == 1)  rpaddle.d = 0;
    else if (rpaddle.y <= 2*paddle.w && rpaddle.d == -1)                   rpaddle.d = 0;
    OLED_fillRect(rpaddle.x, rpaddle.y, paddle.w, paddle.h, fgcolor);
}

void calc_target_y(void)
{
    short  target_x, reflections, y;

    if (ball.dx == 1)  target_x = display.w - ball.w;
    else               target_x = -1 * (display.w - ball.w);
    y = absolute(target_x * (ball.dy / ball.dx) + ball.y);
    reflections = floor(y / display.h);
    if (reflections % 2 == 0)  target_y = y % display.h;
    else                       target_y = display.h - (y % display.h);
    if (time.newminute && ball.dx == -1) {
        if (target_y > (display.h - paddle.h))  target_y = target_y - paddle.h;
        else                                    target_y = target_y + paddle.h;
    } else if (time.newhour && ball.dx == 1) {
        if (target_y > (display.h - paddle.h))  target_y = target_y - paddle.h;
        else                                    target_y = target_y + paddle.h;
    }
}

void changescore(int newgame)
{
    score(bgcolor);
    if (time.newhour) {
        lscore = time.hour;
        rscore = time.minute;
        time.newhour = 0;
        time.newminute = 0;
    } else if (time.newminute) {
        rscore = time.minute;
        time.newminute = 0;
    }
    if (newgame) {
        pause   = 100;
        ball.x  = paddle.w;
        ball.y  = lpaddle.y + (paddle.h / 2);
        ball.dx = 1;
        ball.dy = (ball.dy > 0)? -1: 1;
        calc_target_y();
    }
}

void runball(void)
{
    if (pause > 0) { 
        pause = pause - 1; 
        return; 
    } 
    OLED_fillRect(ball.x, ball.y, ball.w, ball.h, bgcolor);

    score(fgcolor);

    ball.x = ball.x + ball.dx;
    ball.y = ball.y + ball.dy;
    if (ball.dx == -1 && ball.x == paddle.w && ball.y + ball.h >= lpaddle.y && ball.y <= lpaddle.y + paddle.h) {
        ball.dx = ball.dx * -1;
        calc_target_y(); 
    } else if (ball.dx == 1 && ball.x + ball.w == display.w - paddle.w && ball.y + ball.h >= rpaddle.y && ball.y <= rpaddle.y + paddle.h) {
        ball.dx = ball.dx * -1;
        calc_target_y();
    } else if ((ball.dx == 1 && ball.x >= display.w) || (ball.dx == -1 && ball.x + ball.w < 0)) {
        changescore(1);
    }
    if (ball.y > display.h - ball.w - paddle.w - 1 || ball.y < paddle.w +1) {
        ball.dy = ball.dy * -1;
    }
    OLED_fillRect(ball.x, ball.y, ball.w, ball.h, fgcolor);
} 

void initgame(void)
{
    lpaddle.y = random(0, display.h - paddle.h);
    rpaddle.y = random(0, display.h - paddle.h);
    ball.y    = lpaddle.y + (paddle.h / 2);
    lscore    = time.hour;
    rscore    = time.minute;
    calc_target_y();
}

/*
************************************************************
*                    init the pong clock                   *
************************************************************
*/
void clock_pong_init(void)
{
    initgame();
    OLED_Clear(0);
    OLED_Line(0, 0, DISPLAY_X, 0, fgcolor);
    OLED_Line(0, DISPLAY_Y, DISPLAY_X, DISPLAY_Y, fgcolor);
    OLED_Text(23, DISPLAY_Y + 3, font_6_8, datetime);
}

/*
************************************************************
*               next loop step of pong clock               *
************************************************************
*/
void clock_pong(void)
{
    leftpaddle();
    rightpaddle();
    runball();
    midline();
//    score(fgcolor);
    OLED_Text(23, DISPLAY_Y + 3, font_6_8, datetime);
    OLED_Update();
    BSP_delay(1);
}

/************************* END ****************************/
