/***********************************************************
*                        DCF-clock                         *
*            a tiny DCF controlled digital clock           *
*                                                          *
*           by Frank Goetze  -  www.embedded-os.de         *
************************************************************
* Permission to use, copy, modify, and distribute this     *
* software in source and binary forms and its              *
* documentation for any purpose and without fee is hereby  *
* granted, provided that the above authors notice appear   *
* in all copies and that both that authors notice and this *
* permission notice appear in supporting documentation.    *
*                                                          *
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND   *
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT    *
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY    *
* AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.     *
* IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR             *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,    *
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF     *
* USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER *
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN        *
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING           *
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE   *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY *
* OF SUCH DAMAGE.                                          *
************************************************************
*                     clock_tetris.c                       *
*    a digital clock with tetris like animated changes     *
************************************************************
*  tetris clock bases on the javascript code from Long Ho  *
*        at: https://github.com/longlho/tetris-clock       *
************************************************************
* Revision-Info:                                           *
*                                                          *
* 1.00a     first alpa-release                             *
*                                                          *
***********************************************************/
#include <samd21.h>
#include <string.h>

#include "../inc/clock_time.h"
#include "../inc/clock_display.h"
#include "../display/OLED_SSD1306.h"
#include "../BSP/clock_Adafruit_Feather_M0.h"

extern time_t  time;
extern unsigned int sec100;
extern char datetime[16];

/**************** helping vars & code *********************/
static long absolute(long value)
{
    if (value < 0) return -value;
    else           return value;  
}

const unsigned char shapes_org[7][4][2] = {
    {{0, 0}, {0, 1}, {0, 2}, {0, 3}},   // 'I'
    {{0, 0}, {0, 1}, {0, 2}, {1, 2}},   // 'L'
    {{1, 0}, {1, 1}, {1, 2}, {0, 2}},   // 'J'
    {{0, 0}, {0, 1}, {1, 0}, {1, 1}},   // 'O'
    {{1, 0}, {2, 0}, {1, 1}, {0, 1}},   // 'S'
    {{0, 1}, {1, 1}, {2, 1}, {1, 0}},   // 'T'
    {{0, 0}, {1, 0}, {1, 1}, {2, 1}},   // 'Z'
};

typedef struct {
    unsigned char shape[4][2];
    int           x,y;
    unsigned char type;
} curr_block_t;

static curr_block_t cShape;

void deleteShape(void)
{
    unsigned char i, j, x, y;

    for (i = 0; i < 4; i++) {
        for (j = 0; j < SHAPE_SIZE; j++) {
            for (y = 0; y < (SHAPE_SIZE - SHAPE_SPARE); y++) {
                for (x = 0; x < (SHAPE_SIZE - SHAPE_SPARE); x++) {
                    OLED_setPixel(cShape.x + cShape.shape[i][0] * SHAPE_SIZE + x, cShape.y + cShape.shape[i][1] * SHAPE_SIZE + y, bgcolor);
                }
            }
        }
    }
}

void printShape(void)
{
    unsigned char i, j, x, y;

    for (i = 0; i < 4; i++) {
        for (j = 0; j < SHAPE_SIZE; j++) {
            for (y = 0; y < (SHAPE_SIZE - SHAPE_SPARE); y++) {
                for (x = 0; x < (SHAPE_SIZE - SHAPE_SPARE); x++) {
                    OLED_setPixel(cShape.x + cShape.shape[i][0] * SHAPE_SIZE + x, cShape.y + cShape.shape[i][1] * SHAPE_SIZE + y, fgcolor);
                }
            }
        }
    }
    OLED_Update();
    BSP_delay(5);
}

void createShape(unsigned char type, unsigned int x, unsigned int y)
{
    switch (type) {
      case 'I':
        memcpy(cShape.shape, shapes_org[0], sizeof(cShape.shape));
        break;
      case 'L':
        memcpy(cShape.shape, shapes_org[1], sizeof(cShape.shape));
        break;
      case 'J':
        memcpy(cShape.shape, shapes_org[2], sizeof(cShape.shape));
        break;
      case 'O':
        memcpy(cShape.shape, shapes_org[3], sizeof(cShape.shape));
        break;
      case 'S':
        memcpy(cShape.shape, shapes_org[4], sizeof(cShape.shape));
        break;
      case 'T':
        memcpy(cShape.shape, shapes_org[5], sizeof(cShape.shape));
        break;
      case 'Z':
        memcpy(cShape.shape, shapes_org[6], sizeof(cShape.shape));
        break;
      default:
        memset(&cShape, 0, sizeof(cShape));
        return;
    }
    cShape.x = x; cShape.y = y;
    cShape.type = type;
    printShape();
}

void moveShape(int xd, int yd)
{
    while((xd | yd) != 0) {
        deleteShape();
        if (absolute(xd) > 0) { cShape.x += (xd > 0)? SHAPE_SIZE : -SHAPE_SIZE; xd += ((xd > 0)? -1 : 1); }
        else if (yd > 0) { cShape.y += SHAPE_SIZE; yd--; }
        printShape();
    }
}

void rotateShape(unsigned char addtimes)
{
    unsigned char tmp, i;

    addtimes++;
    do {
        deleteShape();
        for (i = 0; i < 4; i++) {
            tmp = cShape.shape[i][0];
            cShape.shape[i][0] = cShape.shape[i][1];
            cShape.shape[i][1] = (cShape.type == 'I')? tmp : 2 - tmp;
        }
        addtimes--;
        printShape();
    } while(addtimes != 0);
}

void makeDigit(unsigned char digit, unsigned int x, unsigned int y)
{
    switch (digit) {
      case 0:
        createShape('I', x, y); moveShape(0, 7); rotateShape(0); moveShape(0, 8);
        createShape('I', x, y); moveShape(5, 12);
        createShape('T', x, y); moveShape(0, 13);
        createShape('Z', x, y); moveShape(0, 3); rotateShape(0); moveShape(0, 8);
        createShape('T', x, y); moveShape(0, 3); rotateShape(0); moveShape(3, 10);
        createShape('Z', x, y); moveShape(0, 3); rotateShape(0); moveShape(0, 6);
        createShape('Z', x, y); moveShape(0, 3); rotateShape(0); moveShape(4, 7);
        createShape('T', x, y); moveShape(0, 4); rotateShape(0); moveShape(-1, 1); rotateShape(1); moveShape(0, 2);
        createShape('Z', x, y); moveShape(0, 6);
        createShape('T', x, y); moveShape(0, 1); rotateShape(0); moveShape(0, 1); rotateShape(1); moveShape(3, 6);
        createShape('J', x, y); moveShape(1, 1); rotateShape(0); moveShape(2, 5);
        createShape('I', x, y); moveShape(0, 1); rotateShape(0); moveShape(2, 5);
        break;
      case 1:
        createShape('L', x, y); moveShape(2, 13);
        createShape('L', x, y); moveShape(0, 7); rotateShape(1); moveShape(1, 5);
        createShape('O', x, y); moveShape(2, 10);
        createShape('L', x, y); moveShape(2, 7);
        createShape('L', x, y); moveShape(0, 4); rotateShape(1); moveShape(1, 2);
        break;
      case 2:
        createShape('I', x, y); moveShape(1, 10); rotateShape(0); moveShape(0, 5);
        createShape('J', x, y); moveShape(3, 9); rotateShape(0); moveShape(0, 4);
        createShape('I', x, y); moveShape(0, 12);
        createShape('L', x, y); moveShape(1, 12);
        createShape('I', x, y); moveShape(1, 6); rotateShape(0); moveShape(0, 5);
        createShape('L', x, y); moveShape(0, 6); rotateShape(0); moveShape(0, 1); rotateShape(0); moveShape(0, 1); rotateShape(0); moveShape(0, 2);
        createShape('J', x, y); moveShape(3, 5); rotateShape(0); moveShape(0, 4);
        createShape('O', x, y); moveShape(4, 8);
        createShape('J', x, y); moveShape(3, 5); rotateShape(0);
        createShape('I', x, y); rotateShape(0); moveShape(-1, 7); moveShape(2, 0);
        createShape('L', x, y); rotateShape(1); rotateShape(0); moveShape(0, 6);
        break;
      case 3:
        createShape('J', x, y); moveShape(0, 3); rotateShape(0); moveShape(2, 3); rotateShape(0); moveShape(0, 5); rotateShape(0); moveShape(0, 3);
        createShape('O', x, y); moveShape(0, 14);
        createShape('J', x, y); moveShape(3, 12);
        createShape('I', x, y); moveShape(5, 12);
        createShape('I', x, y); moveShape(0, 4); rotateShape(0); moveShape(1, 7);
        createShape('L', x, y); moveShape(0, 4); rotateShape(0); moveShape(0, 2); rotateShape(1); moveShape(0, 4);
        createShape('J', x, y); moveShape(0, 4); rotateShape(0); moveShape(3, 5);
        createShape('O', x, y); moveShape(4, 8);
        createShape('I', x, y); moveShape(0, 2); rotateShape(0); moveShape(1, 5);
        createShape('L', x, y); moveShape(0, 2); rotateShape(0); moveShape(0, 2); rotateShape(1); moveShape(0, 2);
        createShape('J', x, y); moveShape(0, 2); rotateShape(0); moveShape(3, 3);
        break;
      case 4:
        createShape('O', x, y); moveShape(4, 14);
        createShape('O', x, y); moveShape(4, 12);
        createShape('I', x, y); moveShape(0, 4); rotateShape(0); moveShape(1, 7);
        createShape('L', x, y); moveShape(0, 4); rotateShape(0); moveShape(0, 2); rotateShape(1); moveShape(0, 4);
        createShape('J', x, y); moveShape(0, 4); rotateShape(0); moveShape(3, 5);
        createShape('I', x, y); moveShape(4, 6);
        createShape('J', x, y); moveShape(0, 7);
        createShape('I', x, y); moveShape(5, 6);
        createShape('J', x, y); rotateShape(0); moveShape(0, 1); rotateShape(0); moveShape(-1, 5);
        break;
      case 5:
        createShape('J', x, y); rotateShape(0); moveShape(2, 2); rotateShape(0); moveShape(0, 10); rotateShape(0); moveShape(0, 2);
        createShape('O', x, y); moveShape(0, 14);
        createShape('I', x, y); moveShape(5, 12);
        createShape('J', x, y); moveShape(3, 12);
        createShape('I', x, y); moveShape(0, 4); rotateShape(0); moveShape(1, 7);
        createShape('L', x, y); moveShape(0, 4); rotateShape(0); moveShape(0, 2); rotateShape(1); moveShape(0, 4);
        createShape('O', x, y); moveShape(0, 8);
        createShape('J', x, y); moveShape(0, 4); rotateShape(0); moveShape(3, 5);
        createShape('I', x, y); moveShape(0, 2); rotateShape(0); moveShape(1, 5);
        createShape('J', x, y); moveShape(0, 2); rotateShape(0); moveShape(3, 3);
        createShape('L', x, y); moveShape(0, 2); rotateShape(0); moveShape(0, 2); rotateShape(1); moveShape(0, 2);
        break;
      case 6:
        createShape('J', x, y); rotateShape(0); moveShape(0, 2); rotateShape(0); moveShape(0, 10); rotateShape(0); moveShape(0, 2);
        createShape('T', x, y); moveShape(3, 14);
        createShape('S', x, y); moveShape(1, 1); rotateShape(0); moveShape(3, 11);
        createShape('J', x, y); rotateShape(0); moveShape(1, 1); rotateShape(1); moveShape(0, 12);
        createShape('T', x, y); moveShape(1, 1); rotateShape(0); moveShape(3, 9);
        createShape('S', x, y); moveShape(2, 10);
        createShape('J', x, y); rotateShape(0); moveShape(0, 1); rotateShape(0); moveShape(0, 9);
        createShape('I', x, y); moveShape(0, 10);
        createShape('L', x, y); moveShape(0, 7);
        createShape('L', x, y); moveShape(0, 2); rotateShape(1); moveShape(-1, 4);
        break;
      case 7:
        createShape('O', x, y); moveShape(4, 14);
        createShape('L', x, y); moveShape(4, 11);
        createShape('L', x, y); moveShape(4, 7); rotateShape(1); moveShape(-1, 3);
        createShape('O', x, y); moveShape(4, 8);
        createShape('I', x, y); moveShape(0, 2); rotateShape(0); moveShape(1, 5);
        createShape('J', x, y); moveShape(0, 2); rotateShape(0); moveShape(3, 3);
        createShape('L', x, y); moveShape(0, 1); rotateShape(0); moveShape(0, 1); rotateShape(1); moveShape(0, 4);
        break;
      case 8:
        createShape('L', x, y); moveShape(3, 12); rotateShape(0); moveShape(0, 1);
        createShape('J', x, y); moveShape(0, 9); rotateShape(0); moveShape(0, 1); rotateShape(1); moveShape(0, 4);
        createShape('S', x, y); moveShape(3, 13);
        createShape('Z', x, y); moveShape(0, 13);
        createShape('L', x, y); moveShape(0, 10);
        createShape('J', x, y); moveShape(4, 10);
        createShape('I', x, y); moveShape(0, 2); rotateShape(0); moveShape(1, 9);
        createShape('L', x, y); moveShape(2, 6); rotateShape(0); moveShape(0, 2);
        createShape('T', x, y); moveShape(0, 3); rotateShape(0); moveShape(0, 5);
        createShape('T', x, y); moveShape(1, 6);
        createShape('J', x, y); moveShape(0, 3); rotateShape(1); moveShape(-1, 3);
        createShape('I', x, y); moveShape(5, 6);
        createShape('L', x, y); moveShape(1, 1); rotateShape(0); moveShape(1, 1); rotateShape(0); moveShape(0, 4);
        break;
      case 9:
        createShape('L', x, y); moveShape(4, 13);
        createShape('L', x, y); moveShape(4, 7); rotateShape(1); moveShape(-1, 5);
        createShape('I', x, y); moveShape(1, 2); rotateShape(0); moveShape(0, 9);
        createShape('T', x, y); moveShape(0, 3); rotateShape(0); moveShape(0, 2); rotateShape(1); moveShape(-1, 4);
        createShape('L', x, y); moveShape(0, 3); rotateShape(0); moveShape(2, 5);
        createShape('S', x, y); moveShape(0, 3); rotateShape(0); moveShape(0, 4);
        createShape('I', x, y); moveShape(5, 8);
        createShape('J', x, y); moveShape(0, 3); rotateShape(0); moveShape(2, 3);
        createShape('J', x, y); moveShape(0, 3); rotateShape(0); moveShape(3, 2);
        createShape('T', x, y); moveShape(0, 2); rotateShape(0); moveShape(0, 1); rotateShape(0); moveShape(0, 2);
        break;
      default:
        createShape('O', x, y); moveShape(2, 12);
        createShape('O', x, y); moveShape(2, 8);
        break;
    }
}

#define POS_DS  (8 * SHAPE_SIZE)
#define POS_TC  (OLED_MAX_X/2 - (2*POS_DS + 8))
#define inv_t_c (unsigned char)-1
const time_t  invalid_time = { inv_t_c, inv_t_c, inv_t_c, inv_t_c, inv_t_c };
volatile time_t tetris_time;
/*
************************************************************
*                   init the tetris clock                  *
************************************************************
*/
void clock_tetris_init(void)
{
    OLED_Clear(0);
    memcpy((void *)&tetris_time, (void *)&invalid_time, sizeof(time_t));
    OLED_Line(0, DISPLAY_Y, DISPLAY_X, DISPLAY_Y, fgcolor);
    OLED_Text(23, DISPLAY_Y + 3, font_6_8, datetime);
    OLED_Update();
}

/*
************************************************************
*              next loop step of tetris clock              *
************************************************************
*/
void clock_tetris(void)
{
    unsigned short s = time.second;
    unsigned char d = 0;

    if (tetris_time.minute == inv_t_c) d = 0x1F;
    else {
        if ((tetris_time.hour / 10)   != (time.hour / 10))   { OLED_tetris_clearDigit(POS_TC, 0); if ((time.hour / 10) > 0) d |= 0x01; }
        if ((tetris_time.hour % 10)   != (time.hour % 10))   { OLED_tetris_clearDigit(POS_TC + 1*POS_DS, 0); d |= 0x02; }
        if ((tetris_time.minute / 10) != (time.minute / 10)) { OLED_tetris_clearDigit(POS_TC + 3*POS_DS, 0); d |= 0x08; }
        if ((tetris_time.minute % 10) != (time.minute % 10)) { OLED_tetris_clearDigit(POS_TC + 4*POS_DS, 0); d |= 0x10; }
    }
    if (d & 0x01) if ((time.hour / 10) > 0) 
                  makeDigit(time.hour / 10  , POS_TC           , 0); 
    if (d & 0x02) makeDigit(time.hour % 10  , POS_TC + 1*POS_DS, 0);
    if (d & 0x04) makeDigit(':'             , POS_TC + 2*POS_DS +1, 0);
    if (d & 0x08) makeDigit(time.minute / 10, POS_TC + 3*POS_DS, 0);
    if (d & 0x10) makeDigit(time.minute % 10, POS_TC + 4*POS_DS, 0);
    memcpy((void *)&tetris_time, (void *)&time, sizeof(time_t));
    OLED_Text(23, DISPLAY_Y + 3, font_6_8, datetime);
    OLED_Update();
    while ((s == time.second) && !(Button_Get())) {
        __NOP();
    }
}

/************************* END ****************************/
