/***********************************************************
*                        DCF-clock                         *
*            a tiny DCF controlled digital clock           *
*                                                          *
*           by Frank Goetze  -  www.embedded-os.de         *
************************************************************
* Permission to use, copy, modify, and distribute this     *
* software in source and binary forms and its              *
* documentation for any purpose and without fee is hereby  *
* granted, provided that the above authors notice appear   *
* in all copies and that both that authors notice and this *
* permission notice appear in supporting documentation.    *
*                                                          *
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND   *
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT    *
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY    *
* AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.     *
* IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR             *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,    *
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF     *
* USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER *
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN        *
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING           *
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE   *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY *
* OF SUCH DAMAGE.                                          *
************************************************************
*                         main.c                           *
*         the entry point & core of the application        *
************************************************************
* Revision-Info:                                           *
*                                                          *
* 1.00a     first alpa-release                             *
*                                                          *
***********************************************************/
#include <samd21.h>
#include <math.h>

#include "../DCF/DCF77_decoder.h"
#include "../inc/clock_display.h"
#include "../display/OLED_SSD1306.h"
#include "../display/icons/clock_icons.c"
#include "../BSP/clock_Adafruit_Feather_M0.h"

/************* the supported clock modes ******************/

extern void clock_simple_init(void);
extern void clock_simple(void);

extern void clock_circle_init(void);
extern void clock_circle(void);

extern void clock_analog_init(void);
extern void clock_analog(void);

extern void clock_big_init(void);
extern void clock_big(void);

extern void clock_pong_init(void);
extern void clock_pong(void);

extern void clock_tetris_init(void);
extern void clock_tetris(void);

typedef struct clock {
    char *name;
    void(*init_Fct)(void); 
    void(*loop_Fct)(void); 
} clock_t;

#ifndef NULL
 #define NULL (void *)0
#endif
clock_t clocks[] = {
    " simple  ", clock_simple_init, clock_simple,
    " circle  ", clock_circle_init, clock_circle,
    " analog  ", clock_analog_init, clock_analog,
    "super big", clock_big_init,    clock_big,
    "  pong   ", clock_pong_init,   clock_pong,
    " tetris  ", clock_tetris_init, clock_tetris,
    NULL,        NULL,              NULL        // end mark (required!)
};

/**************** helping vars & code *********************/
static const int  monthDays[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
static const char *wday_str[] = {"--","Mo","Di","Mi","Do","Fr","Sa","So"};
char datetime[16] = "               ";

// time here to handle global over all clock modes and sync with DCF
time_t  time = { CURR_HOUR, CURR_MIN, 0, 0, 0, 0, 0, 0, 0};
unsigned int sec100 = 0;

static void t_itoa(char *buff, unsigned int value, unsigned int s_div, unsigned char mode)
{
    if ((value / s_div) > 0) { *buff++ = (value / s_div) + '0'; value %= s_div; }
    else if (mode != 0) *buff++ = '0';
    *buff++ = value + '0';
    *buff++ = 0;
}

void SysTick_Handler_time(void) {
    if (++sec100 >= 100) { 
        time.second++;
        sec100 = 0;
    }
    DCF_Handler_time();
}

static void calctime(void)
{
    if (time.second > 59) {
        time.newminute = 1;
        time.second = 0;
        time.minute++;
    }
    if (time.minute > 59) {
        time.newminute = 0;
        time.newhour = 1;
        time.minute = 0;
        time.hour++;
    }
    if (time.hour > 23) {
        time.hour = 0;
        time.day++;
        if (time.wday > 0) time.wday++;
        if (time.wday > 7) time.wday = 1;
    }
    if (time.day > monthDays[time.month - 1]) {
        if ((time.month == 2) && ((time.year % 4) == 0) && (time.day <= (monthDays[time.month - 1] + 1))) {
        } else {    
            time.day = 1;
            time.month++;
        }
    }
    if (time.month > 12) {
        time.month = 1;
        time.year++;
    }
    DCF_sync_Time(&time);
}

static void calcday(void)
{
    datetime[0] = wday_str[time.wday][0];
    datetime[1] = wday_str[time.wday][1];
    datetime[2] = ',';
    t_itoa(&datetime[3], time.day, 10, 1);   datetime[5] = '.';
    t_itoa(&datetime[6], time.month, 10, 1); datetime[8] = '.';
    datetime[9] = '2'; datetime[10] = '0';
    t_itoa(&datetime[11], time.year, 10, 1); datetime[13] = '\0';
}

unsigned char volatile rx_det = 0;
unsigned char dcf_det = 0;
static void dcf_check(unsigned char mode)
{
    unsigned char *icon;
    
    if ((mode != 0) || (dcf_det != rx_det)) {
        switch (rx_det) {
            case 1:  icon = (unsigned char *)DCF_Symbol_collect; break;
            case 2:  icon = (unsigned char *)DCF_Symbol_full; break;
            default: icon = (unsigned char *)DCF_Symbol_no; break;
        }
        OLED_Image(003, DISPLAY_Y + 1, icon, DCF_Symbol_X, DCF_Symbol_Y, 0);
        dcf_det = rx_det;
    }
}

float vbat_old = 0;
static void vbat_check(unsigned char mode)
{
    unsigned char *icon;
    float vbat = ADC_VBATread();
    
    if ((mode != 0) || (fabs(vbat - vbat_old) > 0.02)) {
/*
        if (vbat > 4.2) {
            icon = (unsigned char *)Battery_no;
        } else 
*/
        if (vbat > 3.8) {
            icon = (unsigned char *)Battery_charge;
        } else if (vbat > 3.65) {
            icon = (unsigned char *)Battery_full;
        } else if (vbat > 3.5) {
            icon = (unsigned char *)Battery_66;
        } else if (vbat > 3.35) {
            icon = (unsigned char *)Battery_33;
        } else
            icon = (unsigned char *)Battery_00;
        OLED_Image(110, DISPLAY_Y + 3, icon, Battery_X, Battery_Y, 0);
        vbat_old = vbat;
    }
}

void clock_selector(unsigned char *clock)
{
    unsigned char i = 0; 
    
    do {
        while (Button_Get()) __NOP();
        LED_On(); OLED_Clear(0); i = 0;
        while(clocks[i].name != NULL) { OLED_Text(32, 2 + i * 10, font_8_8, clocks[i].name); i++; }
        OLED_Rect(30, 1 + *clock * 10, 105, 11 + *clock * 10, 1);
        OLED_Update(); BSP_delay(100); LED_Off();
        if (Button_Get()) { *clock += 1; if (clocks[*clock].name == NULL) *clock = 0; }
    } while (Button_Get());
}

/*
************************************************************
*          the MAIN entry & loop of the DCF-clock          *
************************************************************
*/
int main(void)
{
    unsigned char clock = 0;

    SystemInit();
    SetupMainClock();
    SystemCoreClockUpdate();
    if (SysTick_Config(SystemCoreClock / 100)) {
        while (1) __NOP();
    }

    BSP_Init();

    OLED_Init();
    OLED_Logo();

    clocks[clock].init_Fct();
    vbat_check(1);
    dcf_check(1);
    OLED_Update(); 

    while (1) {
        if (Button_Get()) {
            clock_selector(&clock);
            clocks[clock].init_Fct();
            vbat_check(1);
            dcf_check(1);
            OLED_Update(); 
        }
        calctime();
        calcday();
        vbat_check(0);
        dcf_check(0);
        clocks[clock].loop_Fct();
    }
}

/************************* END ****************************/
