/***********************************************************
*                        DCF-clock                         *
*            a tiny DCF controlled digital clock           *
*                                                          *
*           by Frank Goetze  -  www.embedded-os.de         *
************************************************************
* Permission to use, copy, modify, and distribute this     *
* software in source and binary forms and its              *
* documentation for any purpose and without fee is hereby  *
* granted, provided that the above authors notice appear   *
* in all copies and that both that authors notice and this *
* permission notice appear in supporting documentation.    *
*                                                          *
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND   *
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT    *
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY    *
* AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.     *
* IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR             *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,    *
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF     *
* USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER *
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN        *
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING           *
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE   *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY *
* OF SUCH DAMAGE.                                          *
************************************************************
*                     OLED_SSD1306.c                       *
*                   the display driver                     *
************************************************************
* Revision-Info:                                           *
*                                                          *
* 1.00a     first alpa-release                             *
*                                                          *
***********************************************************/
#include <samd21.h>

#include "ascii/clock_ascii.h"

#include "../display/OLED_SSD1306.h"
#include "../BSP/clock_Adafruit_Feather_M0.h"

/************* commands SSD1306 controller ****************/
#define OLED_SET_COL_HI     0x10
#define OLED_SET_COL_LO     0x00
#define OLED_SET_LINE       0x40
#define OLED_SET_CONTRAST   0x81
#define OLED_SET_BRIGHTNESS 0x82
#define OLED_SET_LOOKUP     0x91
#define OLED_SET_SEG_REMAP0 0xA0
#define OLED_SET_SEG_REMAP1 0xA1
#define OLED_EON_OFF        0xA4
#define OLED_EON_ON         0xA5
#define OLED_DISP_NOR       0xA6
#define OLED_DISP_REV       0xA7
#define OLED_MULTIPLEX      0xA8
#define OLED_MASTER_CONFIG  0xAD
#define OLED_CHARGE_PUMP    0x8D
#define OLED_PUMP_OFF       0x10
#define OLED_PUMP_ON        0x14
#define OLED_DISP_OFF       0xAE
#define OLED_DISP_ON        0xAF
#define OLED_SET_PAGE       0xB0
#define OLED_SET_SCAN_FLIP  0xC0
#define OLED_SET_SCAN_NOR   0xC8
#define OLED_SET_OFFSET     0xD3
#define OLED_SET_RATIO_OSC  0xD5
#define OLED_SET_AREA_COLOR 0xD8
#define OLED_SET_CHARGE     0xD9
#define OLED_SET_PADS       0xDA
#define OLED_SET_VCOM       0xDB
#define OLED_NOP            0xE3
#define OLED_SCROLL_RIGHT   0x26
#define OLED_SCROLL_LEFT    0x27
#define OLED_SCROLL_VR      0x29
#define OLED_SCROLL_VL      0x2A
#define OLED_SCROLL_OFF     0x2E
#define OLED_SCROLL_ON      0x2F
#define OLED_SCROLL_ON      0x2F
#define OLED_VERT_SCROLL_A  0xA3
#define OLED_MEM_ADDRESSING 0x20
#define OLED_SET_COL_ADDR   0x21
#define OLED_SET_PAGE_ADDR  0x22 


/************ the external display buffer *****************/
#define DISPLAY_DATA_SIZE   1024 
#define OLED_MAX_X           128
#define OLED_MAX_Y            64 
typedef struct {
    unsigned char  display_setup[3];
    unsigned char  display_data[DISPLAY_DATA_SIZE];
} Disp_data; 
Disp_data Disp_send; 

static void OLED_SendCmd(unsigned char cmd)
{
#ifdef OLED_SPI
    SPI_CS_en();
    SPI_DC_en();
    SPI_write(cmd);
    SPI_DC_dis();
    SPI_CS_dis();
#elif defined OLED_I2C
    I2C_send(&cmd, 1);
#else
 #warning "unknown display type!"
#endif
}

static void OLED_SendCmdData(unsigned char *cmd, unsigned char clen, unsigned char *data, unsigned short dlen)
{
    unsigned short   i = 0;
    
    if (data == NULL) return;
#ifdef OLED_SPI
    SPI_CS_en();
    if (cmd != NULL) {
        SPI_DC_en();
        while (i < clen) SPI_write(cmd[i++]);
        i = 0;
        SPI_DC_dis();
    }
    if (data != NULL) {
        while (i < dlen) SPI_write(data[i++]);
    }
    SPI_CS_dis();
#elif defined OLED_I2C
    if (cmd != NULL) {
        while (i < clen) I2C_send(&(cmd[i++]), 1);
    }
    if (data != NULL) {
        I2C_send(data, dlen);
    }
#else
 #warning "unknown display type!"
#endif
}

static void OLED_Init_SSD1306(void)
{
#ifdef OLED_SPI
    BSP_delay(10); 
    SPI_RES_en(); BSP_delay(10); SPI_RES_dis(); 
    BSP_delay(10);
    OLED_SendCmd(OLED_DISP_OFF);
    OLED_SendCmd(OLED_SET_RATIO_OSC);
    OLED_SendCmd(0xA0);
    OLED_SendCmd(OLED_MULTIPLEX);
    OLED_SendCmd(0x3F);
    OLED_SendCmd(OLED_SET_OFFSET);
    OLED_SendCmd(0x00);
    OLED_SendCmd(OLED_SET_LINE);
    OLED_SendCmd(OLED_SET_SEG_REMAP1);
    OLED_SendCmd(OLED_SET_SCAN_NOR);
    OLED_SendCmd(OLED_SET_COL_ADDR);
    OLED_SendCmd(0);
    OLED_SendCmd(127);
    OLED_SendCmd(OLED_SET_AREA_COLOR);
    OLED_SendCmd(0x05);
    OLED_SendCmd(OLED_SET_CONTRAST);
    OLED_SendCmd(0x4F);
    OLED_SendCmd(OLED_SET_CHARGE);
    OLED_SendCmd(0x82);
    OLED_SendCmd(OLED_SET_VCOM);
    OLED_SendCmd(0x3C);
    OLED_SendCmd(OLED_EON_OFF);
    OLED_SendCmd(OLED_DISP_NOR);
    OLED_SendCmd(OLED_MEM_ADDRESSING);
    OLED_SendCmd(0x00);
    BSP_delay(25);
    OLED_SendCmd(OLED_DISP_ON);
#elif defined OLED_I2C
    OLED_SendCmd(OLED_DISP_OFF);
    OLED_SendCmd(0x00 | 0x0);
    OLED_SendCmd(0x10 | 0x0);
    OLED_SendCmd(0x40 | 0x0);
    OLED_SendCmd(OLED_SET_CONTRAST);
    OLED_SendCmd(0xDF);
    OLED_SendCmd(OLED_SET_SEG_REMAP1);
    OLED_SendCmd(OLED_SET_SCAN_NOR);
    OLED_SendCmd(OLED_DISP_NOR);
    OLED_SendCmd(OLED_EON_OFF);
    OLED_SendCmd(OLED_MULTIPLEX);
    OLED_SendCmd(OLED_MAX_Y - 1);
    OLED_SendCmd(OLED_SET_OFFSET);
    OLED_SendCmd(0x0);
    OLED_SendCmd(OLED_SET_RATIO_OSC);
    OLED_SendCmd(0x80);
    OLED_SendCmd(OLED_SET_CHARGE);
    OLED_SendCmd(0xF1);
    OLED_SendCmd(OLED_SET_PADS);
    OLED_SendCmd(0x12);
    OLED_SendCmd(OLED_SET_VCOM);
    OLED_SendCmd(0x40);
    OLED_SendCmd(OLED_MEM_ADDRESSING);
    OLED_SendCmd(0x00);
    OLED_SendCmd(OLED_SET_LINE | 0x0); 
    OLED_SendCmd(OLED_CHARGE_PUMP);
    OLED_SendCmd(0x14);
    OLED_SendCmd(OLED_SCROLL_OFF);
    OLED_SendCmd(OLED_EON_OFF);
    OLED_SendCmd(OLED_DISP_ON);
#else
 #warning "unknown display type!"
#endif
    Disp_send.display_setup[0] = OLED_SET_PAGE;
    Disp_send.display_setup[1] = OLED_SET_COL_HI;
    Disp_send.display_setup[2] = OLED_SET_COL_LO;
}

/**************** helping vars & code *********************/
static long absolute(long value) {
    if (value < 0) return -value;
    else return value;  
}

/*
************************************************************
*        write the external buffer to OLED display         *
************************************************************
*/
void OLED_Update(void)
{
    OLED_SendCmdData(&Disp_send.display_setup[0], 3, &Disp_send.display_data[0], DISPLAY_DATA_SIZE);
    OLED_SendCmd(OLED_SET_LINE);
}

/*
************************************************************
*                    Clear display buffer                  *
************************************************************
*/
void OLED_Clear(unsigned char pattern)
{
    unsigned short   i;
    
    for(i=0; i<DISPLAY_DATA_SIZE; i++) Disp_send.display_data[i]=pattern;
    OLED_Update();
} 

/*
************************************************************
*                 init the OLED display                    *
************************************************************
*/
void OLED_Init(void)
{
#ifdef OLED_SPI
    SPI_RES_dis(); SPI_CS_dis(); SPI_DC_dis();  
#endif
    OLED_Init_SSD1306();
    OLED_Clear(0);
}

/*
************************************************************
*                      draw a pixel                        *
************************************************************
*/
void OLED_setPixel(unsigned short x, unsigned short y, unsigned char set)
{
    if ((x >= OLED_MAX_X) || (y >= OLED_MAX_Y)) return;
    if (set != 0)
        Disp_send.display_data[((y << 4) & 0xFF80) + x] |= (unsigned char)(0x01 << (y & 0x07));
    else
        Disp_send.display_data[((y << 4) & 0xFF80) + x] &= ~((unsigned char)(0x01 << (y & 0x07)));
} 

/*
************************************************************
*                        draw a line                       *
************************************************************
*/
void OLED_Line(short x0, short y0, short x1, short y1, unsigned char set)
{
    short dx, dy, inx, iny, e;
     
    dx = x1 - x0; inx = dx > 0 ? 1 : -1;
    dy = y1 - y0; iny = dy > 0 ? 1 : -1;
    dx = absolute(dx);
    dy = absolute(dy);
    if(dx >= dy) {
        dy <<= 1; e = dy - dx; dx <<= 1;
        while (x0 != x1) {
            OLED_setPixel(x0, y0, set);
            if(e >= 0) { y0 += iny; e -= dx; }
            e += dy; x0 += inx;
        }
    } else {
        dx <<= 1; e = dx - dy; dy <<= 1;
        while (y0 != y1) {
            OLED_setPixel(x0, y0, set);
            if(e >= 0) { x0 += inx; e -= dy; }
            e += dx; y0 += iny;
        }
    }
    OLED_setPixel(x0, y0, set);
} 

/*
************************************************************
*                     draw a rectangle                     *
************************************************************
*/
void OLED_Rect(short x0, short y0, short x1, short y1, unsigned char set)
{
    if ((x0 >= OLED_MAX_X) || (y0 >= OLED_MAX_Y) || (x1 >= OLED_MAX_X) || (y1 >= OLED_MAX_Y))
        return;
    OLED_Line(x0, y0, x0, y1, set);
    OLED_Line(x0, y1, x1, y1, set);
    OLED_Line(x1, y0, x1, y1, set);
    OLED_Line(x0, y0, x1, y0, set);
}

void OLED_fillRect(short x0, short y0, short xs, short ys, unsigned char set)
{
     OLED_Rect(x0, y0, x0+xs, y0+ys, set);
}

/*
************************************************************
*                       draw a circle                      *
************************************************************
*/
void OLED_Circle(unsigned short x, unsigned short y, unsigned short rad, unsigned char set)
{
    short xd, yd, err;
    
    if ((x >= OLED_MAX_X) || (y >= OLED_MAX_Y) ||
        ((x + rad) >= OLED_MAX_X) || ((x - rad) >= OLED_MAX_X) ||
        ((y + rad) >= OLED_MAX_Y) || ((y - rad) >= OLED_MAX_Y))
        return;
    xd = rad; yd = 0; err = -rad;
    while (xd >= yd) {
        OLED_setPixel(x + xd, y + yd, set);
        OLED_setPixel(x - xd, y + yd, set);
        OLED_setPixel(x + xd, y - yd, set);
        OLED_setPixel(x - xd, y - yd, set);
        OLED_setPixel(x + yd, y + xd, set);
        OLED_setPixel(x - yd, y + xd, set);
        OLED_setPixel(x + yd, y - xd, set);
        OLED_setPixel(x - yd, y - xd, set);
        err += yd; yd++; err += yd;
        if (err >= 0) { xd--; err -= xd; err -= xd; }
    }
}

/*
************************************************************
*                      write RAW image                     *
************************************************************
*/
void OLED_Image(unsigned short x0, unsigned short y0, unsigned char *img, unsigned short xs, unsigned short ys, unsigned char inv)
{
    unsigned int   i=0, x, y;
    unsigned char  tmp;

    for (y = 0; y < ys; y++) {
        for (x = 0; x < xs; x++) {
            tmp = img[i++];
            if(inv) tmp += 1;  
            if (tmp != 0 ) {
                OLED_setPixel(x0 + x, y0 + y, 1);
            } else {
                OLED_setPixel(x0 + x, y0 + y, 0);
            }
        }
    }
}
    
/*
************************************************************
*                   write char on position                 *
************************************************************
*/
void OLED_Char(unsigned short x0, unsigned short y0, font_t font, char ASCI)
{
    switch (font) {
        case font_8_8: {
                unsigned short i, j;
                unsigned char  buffer[8], tmp_char;

                GetASCIICode(buffer,ASCI);
                for (i=0; i<8; i++) {
                    tmp_char = buffer[i];
                    for (j=0; j<8; j++) {
                        if (((tmp_char >> (7 - j)) & 0x01) == 0x01) {
                            OLED_setPixel(x0 + j, y0 + i, 1);
                        } else {
                            OLED_setPixel(x0 + j, y0 + i, 0);
                        }
                    }
                }
            } 
            break;
        case font_6_8: {
                unsigned short i, j;
                unsigned char  buffer[6], tmp_char;
                
                GetASCIICodeS(buffer,ASCI);
                for (j=0; j<6; j++) {
                    tmp_char = buffer[j];
                    for (i=0; i<8; i++) {
                        if (((tmp_char >> (7 - i)) & 0x01) == 0x01) {
                            OLED_setPixel(x0 + j, y0 + 7 - i, 1);
                        } else {
                            OLED_setPixel(x0 + j, y0 + 7 - i, 0);
                        }
                    }
                }
            }
            break;
        case font_Dot: {
                unsigned short i, j, s = 0, tmp_char;
                unsigned char  buffer[44];
                
                GetASCIICodeD(buffer,ASCI);
                for (i=0; i<22; i++) {
                    tmp_char = (buffer[s] << 8) | buffer[s+1]; s+=2;
                    for (j=0; j<16; j++) {
                        if (((tmp_char >> (15 - j)) & 0x0001) == 0x0001) {
                            OLED_setPixel(x0 + j, y0 + i, 1);
                        } else {
                            OLED_setPixel(x0 + j, y0 + i, 0);
                        }
                    }
                }
            }
            break;
        case font_Dot_small: {
                unsigned short i, j, ji, s, tmp_char;
                unsigned char  buffer[44];
                
                GetASCIICodeD(buffer,ASCI); s = 0;
                for (i=0; i<16; i++) {
                    if (i == 7) s += 6;
                    if ((i == 13) && (ASCI != 0x37)) s+= 6;
                    if ((i == 16) && (ASCI == 0x37)) s+= 6;
                    tmp_char = (buffer[s] << 8) | buffer[s+1]; s+=2;
                    ji = 0;
                    for (j=0; j<16; j++) {
                        if ((ASCI == 0x31) || (ASCI == 0x37)) { 
                            if (j == 0) j += 3;
                        } else if (j == 4) j += 3;
                        if (((tmp_char >> (15 - j)) & 0x0001) == 0x0001) {
                            OLED_setPixel(x0 + ji, y0 + i, 1);
                        } else {
                            OLED_setPixel(x0 + ji, y0 + i, 0);
                        }
                        ji++;
                    }
                }
            }
            break;
        case font_16_segment: {
                unsigned short i, j, s = 0;
                unsigned int   tmp_char;
                unsigned char  buffer[3*36];
                
                GetASCIICode16(buffer,ASCI);
                for (i=0; i<36; i++) {
                    tmp_char = (buffer[s] << 16) | (buffer[s+1] << 8) | buffer[s+2]; s+=3;
                    for (j=0; j<24; j++) {
                        if (((tmp_char >> (23 - j)) & 1) == 1) {
                            OLED_setPixel(x0 + j, y0 + i, 1);
                        } else {
                            OLED_setPixel(x0 + j, y0 + i, 0);
                        }
                    }
                }
            }
            break;
        default:
            break;
    }
}

/*
************************************************************
*                   write text on position                 *
************************************************************
*/
void OLED_Text(unsigned short x0, unsigned short y0, font_t font, char *str)
{
    unsigned char TempChar;
    
    do {
        TempChar = *str++;  
        OLED_Char(x0, y0, font, TempChar);    
        switch (font) {
            case font_8_8:
                if (x0 < OLED_MAX_X - 8) { x0 += 8; } 
                else if (y0 < OLED_MAX_Y - 8) { x0 = 0; y0 += 8; }   
                else { x0 = 0; y0 = 0; }    
                break;
            case font_6_8:
                if (x0 < OLED_MAX_X - 6) { x0 += 6; } 
                else if (y0 < OLED_MAX_Y - 8) { x0 = 0; y0 += 8; }   
                else { x0 = 0; y0 = 0; }    
                break;
            case font_Dot:
                if (x0 < OLED_MAX_X - 17) { x0 += 17; } 
                else if (y0 < OLED_MAX_Y - 22) { x0 = 0; y0 += 22; }   
                else { x0 = 0; y0 = 0; }    
                break;
            case font_Dot_small:
                if (x0 < OLED_MAX_X - 15) { x0 += 15; } 
                else if (y0 < OLED_MAX_Y - 17) { x0 = 0; y0 += 17; }   
                else { x0 = 0; y0 = 0; }    
                break;
            case font_16_segment:
                if (x0 < OLED_MAX_X - 26) { x0 += 26; } 
                else if (y0 < OLED_MAX_Y - 36) { x0 = 0; y0 += 36; }   
                else { x0 = 0; y0 = 0; }    
                break;
            default:
                break;
        }
    } while (*str != 0);
}

/*
************************************************************
*   clear a tetris digit field (Y must be multiple of 8)   *
************************************************************
*/
void OLED_tetris_clearDigit(unsigned short x0, unsigned short y0)
{
    unsigned short i, j;

    if ((x0 >= OLED_MAX_X) || (y0 >= OLED_MAX_Y)) return;
    if ((y0 % 8) > 0) return;
    for (i=0; i<6; i++) {
        for (j=0; j<21; j++) {
            Disp_send.display_data[(( (y0 + (i*8)) << 4) & 0xFF80) + (x0+j)] = 0;
        }
    }
}

/*
************************************************************
*   write "embedded-os.de" logo and wait 3sec or button    *
************************************************************
*/
void OLED_Logo(void)
{
    const unsigned char logo1p[] = {0x00,0x00,0x80,0x40,0x40,0x20,0x20,0x30,0x10,0x10,0x18,0x08,0x08,0x0C,0x0C,0x0C,0x0C,0x04,0x06,0x86,0xC6,0xC6,0x46,0x42,0xC2,0xC2,0xC3,0xC3,0xE3,0xF3,0xF3,
                                    0xE3,0x63,0x73,0x73,0xF3,0xF6,0xE6,0xC6,0xC6,0x86,0x86,0x0E,0x0E,0x0C,0x0C,0x0C,0x0C,0x1C,0x1C,0x38,0x38,0x30,0x70,0x70,0xE0,0xE0,0xC0,0xC0,0x80,0x80,0x00,0x00,0x00,0x00 };
    const unsigned char logo2p[] = {0xE0,0xE0,0x70,0xF8,0xE8,0xE8,0xF8,0xB8,0xB8,0x18,0x1C,0x1E,0x0F,0x0D,0x0D,0x0F,0x0F,0x07,0x07,0x07,0x07,0x03,0x03,0x03,0x01,0x01,0x01,0x00,0x00,0x00,0x00,
                                    0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x81,0x83,0xC3,0xC3,0xC7,0xEF,0xEE,0xEE,0xFC,0xB8,0xB8,0x30,0x70,0x60,0xE0,0x00,0x00,0x00,0x01,0x01,0x03,0x0F,0x1F,0xFF,0xFE,0xFC,0xC0 };
    const unsigned char logo3p[] = {0x03,0x07,0x0C,0x1C,0x18,0x39,0x33,0x63,0xE7,0xC7,0xCF,0x8E,0x1C,0x3C,0x3C,0xF8,0xF8,0xF0,0xF0,0xF0,0xF8,0x38,0x30,0x70,0xF0,0xF8,0xFC,0xFC,0xFC,0xFE,0xF6,
                                    0x06,0x06,0xDE,0xFF,0x7F,0x37,0x37,0x7E,0xFE,0xC0,0xC0,0xE1,0x7F,0x1F,0x07,0x07,0x0F,0x1F,0x38,0x30,0x3F,0x1F,0x80,0xC0,0xC0,0xE0,0xE0,0x70,0x38,0x3E,0x1F,0x0F,0x07,0x00 };
    const unsigned char logo4p[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x03,0x03,0x07,0x06,0x0F,0x0F,0x0F,0x06,0x07,0x1F,0x38,0x30,0x30,0x1F,0x03,0x03,0x01,0x01,0x03,0x07,
                                    0x06,0x06,0x07,0x13,0x10,0x10,0x10,0x10,0x10,0x10,0x18,0x18,0x08,0x08,0x0C,0x0C,0x0C,0x06,0x06,0x07,0x03,0x03,0x03,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };
    const unsigned char logo5p[] = {0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x80,0x00,0x80,0x80,0x80,0x80,0x00,0x00,0xE0,0x00,0x80,0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,
                                    0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0xE0,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0xE0,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0x00,
                                    0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0xE0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,
                                    0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0xE0,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00 };
    const unsigned char logo6p[] = {0x00,0x00,0x1E,0x33,0x70,0x58,0x4C,0x46,0x43,0x21,0x00,0x7F,0x01,0x00,0x00,0x00,0x01,0x7F,0x00,0x00,0x00,0x00,0x7F,0x00,0x7F,0x21,0x40,0x40,0x40,0x40,0x40,0x61,0x3F,0x00,0x1E,0x33,
                                    0x70,0x58,0x4C,0x46,0x43,0x21,0x00,0x1E,0x33,0x40,0x40,0x40,0x40,0x40,0x21,0x7F,0x00,0x1E,0x33,0x40,0x40,0x40,0x40,0x40,0x21,0x7F,0x00,0x1E,0x33,0x70,0x58,0x4C,0x46,0x43,0x21,0x00,
                                    0x1E,0x33,0x40,0x40,0x40,0x40,0x40,0x21,0x7F,0x00,0x00,0x04,0x04,0x04,0x04,0x04,0x00,0x1E,0x21,0x40,0x40,0x40,0x40,0x40,0x21,0x1E,0x00,0x23,0x44,0x4C,0x4C,0x48,0x31,0x00,0x40,0x00,
                                    0x1E,0x33,0x40,0x40,0x40,0x40,0x40,0x21,0x7F,0x00,0x1E,0x33,0x70,0x58,0x4C,0x46,0x43,0x21 };
    unsigned short i;
                                    
    OLED_Clear(0);
    for(i = 0; i < sizeof(logo1p); i++) Disp_send.display_data[(( 8 << 4) & 0xFF80) + 33+i] = logo1p[i];
    for(i = 0; i < sizeof(logo2p); i++) Disp_send.display_data[((16 << 4) & 0xFF80) + 33+i] = logo2p[i];
    for(i = 0; i < sizeof(logo3p); i++) Disp_send.display_data[((24 << 4) & 0xFF80) + 33+i] = logo3p[i];
    for(i = 0; i < sizeof(logo4p); i++) Disp_send.display_data[((32 << 4) & 0xFF80) + 33+i] = logo4p[i];
    for(i = 0; i < sizeof(logo5p); i++) Disp_send.display_data[((48 << 4) & 0xFF80) + i] = logo5p[i];
    for(i = 0; i < sizeof(logo6p); i++) Disp_send.display_data[((56 << 4) & 0xFF80) + i] = logo6p[i];
    OLED_Update();
    BSP_delay(300);
    while (Button_Get()) __NOP();
}

/************************* END ****************************/
