/***********************************************************
* SAM7SExx_DevMaster - Init of external DOGM128-6 on NPCS3 *
*                                                          *
*    generated by Frank Goetze  -   www.embedded-os.de     *
***********************************************************/
#include "../inc/OS_Def_AT91SAM7xxx.h"

#include "GLCDfont5x7.h"

/***********************************************************
*         user-functions prototypes (EA DOG-M 128-6)       *
***********************************************************/
void   GLCD_Init(void);                                    // init the used SPI and the LCD
void   GLCD_Reset(void);                                   // reset the LCD
void   GLCD_OnOff(U08 o);                                  // switch the display ON / OFF
void   GLCD_Contrast(U08 c);                               // set the contrast 0..63
void   GLCD_BackLight(U08 percent);                        // change backlight in %
void   GLCD_ClearScreen(void);                             // clear screen & home
void   GLCD_FlushScreen(void);                             // flush hole buffer into LCD
void   GLCD_UpdateScreen(void);                            // flush only updated(dirty) pages into LCD
void   GLCD_GotoHome(void);                                // set the cursor to home 0,0 (for 5x7 font)
void   GLCD_GotoXY(U08 x, U08 y);                          // set the cursor to 0..127, 0..63 (for 5x7 font)
U16    GLCD_GetXY(void);                                   // get the actual cursor position high-byte=x & low-byte=y
void   GLCD_SetPixel(U08 x, U08 y);                        // set a pixel
void   GLCD_ClearPixel(U08 x, U08 y);                      // clear a pixel
U08    GLCD_GetPixel(U08 x, U08 y);                        // get a pixel (1/0)
void   GLCD_CreateCG(U08 addr, U08 OS_HUGE *array);        // create own character in CGRAM (later)
void   GLCD_PutChar(U08 c, U08 inv);                       // write a char on actual cursor (5x7 font) & update cursor position
void   GLCD_PutString(U08 OS_HUGE *s, U08 inv);            // write a string starting on actual cursor (5x7 font) & update cursor position

/***********************************************************
*                    definitions                           *
***********************************************************/
#define CPU_CRISTAL_CLK       18432000UL                   // processor crystal-clock in Hz

#define AT91S_SPI_NPCS3       AT91S_PB22                   // /MCS3 PIO-pin
#define AT91S_SPI_MISO        AT91S_PIO_PA12               // MISO  PIO-pin
#define AT91S_SPI_MOSI        AT91S_PIO_PA13               // MOSI  PIO-pin
#define AT91S_SPI_SPCK        AT91S_PIO_PA14               // SCK   PIO-pin
#define AT91S_LCD_BL          AT91S_PB25                   // LCD-backlight pin (PWM2)
#define AT91S_LCD_A0          AT91S_PB29                   // A0 or RS (cmd/data) PIO-pin
#define AT91S_LCD_RES         AT91S_PB31                   // LCD-Reset PIO-pin
#define LCDport_CS_ON         AT91S_PIOB_CODR = AT91S_SPI_NPCS3;
#define LCDport_CS_OFF        AT91S_PIOB_SODR = AT91S_SPI_NPCS3;
#define LCDport_A0_LOW        AT91S_PIOB_CODR = AT91S_LCD_A0;
#define LCDport_A0_HIGH       AT91S_PIOB_SODR = AT91S_LCD_A0;

/***********************************************************
*                       globals                            *
***********************************************************/
OS_CONST U08 OS_ROMDATA GLCD_InitParams[] = { 0x40,0xA1,0xC0,0xA6,0xA2,0x2F,0xF8,0x00,0x27,0x81,0x16,0xAC,0x00,0xAF };

U08 GLCDarray[8][128];                                     // display memory (pages)
U08 GLCDpaged;                                             // dirty(updated)-pages
U16 GLCDxy;                                                // cursor-position

/***********************************************************
*          get MCK based on given cristal-CLK              *
***********************************************************/
U32 gl_mck;                                                // global value holder
void OS_getMCK(void)
{
    U32 s_clksrc;
    
    s_clksrc = CPU_CRISTAL_CLK;                                        // set start-clk (cristal-clk) - must be a "define"
    if((AT91S_PMC_MCKR & AT91S_PMC_PLLCLK) == AT91S_PMC_PLLCLK) {      // if PLL is enabled ..
        if((AT91S_CKGR_PLLR & AT91S_CKGR_DIV) > 0) {                   // .. if PLL-divisor set ..
            s_clksrc /= (AT91S_CKGR_PLLR & AT91S_CKGR_DIV);            // div clk with PLL-div
        }
        s_clksrc *= (((AT91S_CKGR_PLLR & AT91S_CKGR_MUL) >> 16) + 1);  // mul clk with PLL-multiplier
    }
    s_clksrc /= (0x0001UL << ((AT91S_PMC_MCKR & AT91C_PMC_PRES_CLK_M) >> 2));  // div clk with MCK-div
    gl_mck   = s_clksrc;                                               // now "gl_mclk" holds the MCK in Hertz
}

/***********************************************************
*       set max possible SPI baudrate and return this      *
***********************************************************/
U32 OS_setSPI_BR(U32 maxclk)
{
    U08 i, m;

    OS_getMCK();
    AT91S_SPI_MR &= ~AT91S_SPI_MR_FDIV;                    // clear FDIV at calc-startup
    for(i=1, m=1; m<32; i=1, m=32) {                       // search lowest clk-div for SPI-speed (highest possible SPI-speed)
        while(((gl_mck / (i * m)) > maxclk) && (i > 0)) i++;
        if(i > 0) break;
    }
    if(m > 1) AT91S_SPI_MR |= AT91S_SPI_MR_FDIV;           // if hole SPI-div greater 255 set FDIV for enable pre-div(32)
    AT91S_SPI_CSR->s[3] = (AT91S_SPI_CSR->s[3] & ~AT91S_SPI_CSR_SCBR_M) | (i << 8);
    return(gl_mck / (i * m));                              // set SPI-div and return the used SPI-speed in Hz
}

/***********************************************************
*                     simple delay                         *
***********************************************************/
void GLCDport_Delay(U32 d)
{
    d *= 10;
    while(d--) { _NOP(); _NOP(); _NOP(); }                 // the delay-times are importend on DOG-M displays !!!
}

/***********************************************************
*                     reset the LCD                        *
***********************************************************/
void GLCD_Reset(void)
{
    AT91S_PIOB_CODR = AT91S_LCD_RES;
    GLCDport_Delay(100);
    AT91S_PIOB_SODR = AT91S_LCD_RES;
    GLCDxy = 0;
}

/***********************************************************
*               write a data-byte to display               *
***********************************************************/
void GLCDport_Byte(U08 w)
{
    LCDport_CS_ON;
    GLCDport_Delay(5);
    AT91S_SPI_TDR = w;                                     // write char and send
    while(!(AT91S_SPI_SR & AT91S_SPI_SR_TDRE));            // wait up to tx-empty
    GLCDport_Delay(50);
    LCDport_CS_OFF;
}

/***********************************************************
*                      write a CMD                         *
***********************************************************/
void GLCDport_Cmd(U08 w)
{
    LCDport_A0_LOW;
    GLCDport_Byte(w);
    LCDport_A0_HIGH;
}

/***********************************************************
*               copy a page to graphic-LCD                 *
***********************************************************/
void GLCDport_copypage(U08 page)
{
    U08 i;
    
    GLCDport_Cmd(0x40);
    GLCDport_Cmd(0xB0 + page);
    GLCDport_Cmd(0x00);
    GLCDport_Cmd(0x10);
    for(i = 0; i < 128; i++) GLCDport_Byte(GLCDarray[page][i]);
    GLCDport_Cmd(0xE3);
    GLCDpaged &= ~(1 << page);
}

/***********************************************************
*          flush hole GLCDarray[] into dislay              *
***********************************************************/
void GLCD_FlushScreen(void)
{
    U08 i;
    
    for(i = 0; i < 8; i++) GLCDport_copypage(i);
}

/***********************************************************
*      flush dirty pages of GLCDarray[] into dislay        *
***********************************************************/
void GLCD_UpdateScreen(void)
{
    U08 i;
    
    for(i = 0; i < 8; i++) if(GLCDpaged & (1 << i)) GLCDport_copypage(i);
}

/***********************************************************
*                     clear screen                         *
***********************************************************/
void GLCD_ClearScreen(void)
{
    U08 i, j;
    
    for(i = 0; i < 8; i++) for(j = 0; j < 128; j++) GLCDarray[i][j] = 0;
    GLCD_FlushScreen();
    GLCDxy = 0;
}

/***********************************************************
*              set cursor position to home                 *
***********************************************************/
void GLCD_GotoHome(void)
{
    GLCDxy = 0;
}

/***********************************************************
*            set cursor position (0..127, 0..63)           *
***********************************************************/
void GLCD_GotoXY(U08 x, U08 y)
{
    if((x < 128) & (y < 64)) GLCDxy = ((U16)(x) << 8) | y;
}

/***********************************************************
*            get cursor position (0..127 & 0..63)          *
***********************************************************/
U16 GLCD_GetXY(void)
{
    return(GLCDxy);
}

/***********************************************************
*                      set a pixel                         *
***********************************************************/
void GLCD_SetPixel(U08 x, U08 y)
{
    if((x < 128) & (y < 64)) {
        GLCDarray[y / 8][x] |= (1 << (y % 8));
        GLCDpaged           |= (1 << (y / 8));
    }
}

/***********************************************************
*                     clear a pixel                        *
***********************************************************/
void GLCD_ClearPixel(U08 x, U08 y)
{
    if((x < 128) & (y < 64)) {
        GLCDarray[y / 8][x] &= ~(1 << (y % 8));
        GLCDpaged           |=  (1 << (y / 8));
    }
}

/***********************************************************
*                   get a pixel (1/0)                      *
***********************************************************/
U08 GLCD_GetPixel(U08 x, U08 y)
{
    if((x < 128) & (y < 64)) {
        return((GLCDarray[y / 8][x] & (1 << (y % 8)))?1:0);
    }
    return(0);
}

/***********************************************************
*                create special character                  *
***********************************************************/
void GLCD_CreateCG(U08 addr, U08 OS_HUGE *array)
{
    // define a special char (later)
}

/***********************************************************
*               write a char to display                    *
***********************************************************/
void GLCD_PutChar(U08 c, U08 inv)
{
    U08 i, x, y, yd, bm;
    
    x = (U08)(GLCDxy >> 8);
    y = (U08)(GLCDxy & 0xFF);
    GLCDpaged |=  (1 << (y / 8));
    for(i = 0; i < 5; i++) {
        if((c >= 0x20) && (c < 0x80)) {
            bm = GLCDfont5x7[5 * (c - 0x20) + i];
        } else {
            // use defined char (later)
            bm = 0;
        }
        if(y % 8) {
            yd = (y / 8) + 1;
            if(yd >= 8) yd -= 8;
        }
        if(!inv) {
            GLCDarray[y / 8][x] |= (bm << (y % 8));
            if(y % 8) GLCDarray[yd][x] |= (bm >> (8 - (y % 8)));
        } else {
            GLCDarray[y / 8][x] &= ~(bm << (y % 8));
            if(y % 8) GLCDarray[yd][x] &= ~(bm >> (8 - (y % 8)));
        }
        x++;
        if(x >= 128) { x -= 128; y += 8; }
        if(y >= 64) y -= 64;
    }
    GLCDpaged |= (1 << (y / 8));
    if(y % 8) GLCDpaged |= (1 << yd);
    x++;
    if(x >= 128) { x -= 128; y += 8; }
    if(y >= 64) y -= 64;
    GLCDxy = ((U16)(x) << 8) | y;
}

/***********************************************************
*               write a string to display                  *
***********************************************************/
void GLCD_PutString(U08 OS_HUGE *s, U08 inv)
{
    U32 i;

    i = 0; while(s[i] != 0) GLCD_PutChar(s[i++], inv);
}

/***********************************************************
*              sets the contrast of display                *
***********************************************************/
void GLCD_Contrast(U08 c)
{
    c &= 0x3F;
    GLCDport_Cmd(0x81);
    GLCDport_Cmd(c);
}

/***********************************************************
*               sets the display ON or OFF                 *
***********************************************************/
void GLCD_OnOff(U08 o)
{
    o = (o > 0)?1:0;
    GLCDport_Cmd(0xAE | o);
}

/***********************************************************
*                 SPI port for LCD init                    *
***********************************************************/
GLCDport_InitSPI(void)
{
    AT91S_PIOB_PER       = AT91S_SPI_NPCS3 | AT91S_LCD_A0 | AT91S_LCD_RES;    // enable GPIO of /CS & A0 & Reset
    AT91S_PIOB_SODR      = AT91S_SPI_NPCS3 | AT91S_LCD_A0 | AT91S_LCD_RES;    // set all high
    AT91S_PIOB_OER       = AT91S_SPI_NPCS3 | AT91S_LCD_A0 | AT91S_LCD_RES;    // output enable
    AT91S_PMC_PCER       = (1 << AT91S_ID_PIOB);                              // enable clock on PIO-B

    AT91S_PIO_PDR        = AT91S_SPI_MISO | AT91S_SPI_MOSI | AT91S_SPI_SPCK;  // disable from PIO-mode
    AT91S_PIO_ASR        = AT91S_SPI_MISO | AT91S_SPI_MOSI | AT91S_SPI_SPCK;  // enable bit-functions
    AT91S_PMC_PCER       = (1 << AT91S_ID_SPI);                               // enable clock on SPI-interface
    AT91S_SPI_CR         = AT91S_SPI_CR_DIS;                                  // disable SPI
    AT91S_SPI_CR         = AT91S_SPI_CR_EN | AT91S_SPI_CR_SWRST;              // enable & reset SPI
    AT91S_SPI_MR         = AT91S_SPI_MR_CS3 | AT91S_SPI_MR_MODFDIS | AT91S_SPI_MR_MSTR;       // CS3 + fault-off + (fix) + master
    AT91S_SPI_CSR->s[3]  = AT91S_SPI_CSR_SCBR_L | AT91S_SPI_CSR_BITS_8 | AT91S_SPI_CSR_CPOL;  // low-speed +  8bit + clk-pol (mode 3)
    AT91S_SPI_CR         = AT91S_SPI_CR_EN;                                   // enable SPI
    OS_setSPI_BR(2000000);                                                    // set max 2MHz
}

/***********************************************************
*               PWM port init (LCD-backlight)              *
***********************************************************/
#define   PWM_PREDIV_VALUE            AT91S_PWM_MCK_16
#define   PWM_PERIOD_VALUE            (gl_mck / ((1 << PWM_PREDIV_VALUE) * 1000))

GLCDport_InitPWM(void)
{   
    OS_getMCK();
    AT91S_PMC_PCER       = (1 << AT91S_ID_PIOB);                              // enable clock on PIO-B (for 0%/100% backlight)
    AT91S_PIOB_PDR       = AT91S_LCD_BL;                                      // disable from PIO-mode
    AT91S_PIOB_ASR       = AT91S_LCD_BL;                                      // enable bit-functions
    AT91S_PWM_DIS        = (1 << AT91S_PWM_CH2);                              // disable PWM2
    AT91S_PMC_PCER       = (1 << AT91S_ID_PWM);                               // enable clock on PWM-interface
    AT91S_PWM_MR         = (PWM_PREDIV_VALUE << AT91S_PWM_MR_PREA_O) | 1;
    AT91S_PWM_CR->CH[AT91S_PWM_CH2].CMR  = PWM_PREDIV_VALUE;                  // CPD is 0 -> CUPD updates the duty-cycle
    AT91S_PWM_CR->CH[AT91S_PWM_CH2].CDTY = PWM_PERIOD_VALUE / 2;              // 50%
    AT91S_PWM_CR->CH[AT91S_PWM_CH2].CPRD = PWM_PERIOD_VALUE;                  // 1kHz
    AT91S_PWM_ENA        = (1 << AT91S_PWM_CH2);                              // enable PWM2
    AT91S_PWM_CR->CH[AT91S_PWM_CH2].CUPD = PWM_PERIOD_VALUE / 2;              // 50% as update value for duty-cycle
}

/***********************************************************
*              graphic-LCD backlight config                *
***********************************************************/
void GLCD_BackLight(U08 percent)
{
    if(percent > 100) percent = 100;
    switch(percent) {
        case 0:                                            // fully OFF
            AT91S_PIOB_PER  = AT91S_LCD_BL;
            AT91S_PIOB_SODR = AT91S_LCD_BL;
            AT91S_PIOB_OER  = AT91S_LCD_BL;
            break;
        case 100:                                          // fully ON
            AT91S_PIOB_PER  = AT91S_LCD_BL;
            AT91S_PIOB_CODR = AT91S_LCD_BL;
            AT91S_PIOB_OER  = AT91S_LCD_BL;
            break;
        default:                                           // 1 ... 99 % (1kHz with "percent" duty-cycle)
            if(AT91S_PIOB_PSR & AT91S_LCD_BL) {            // switch to PWM-mode of needed
                AT91S_PIOB_PDR  = AT91S_LCD_BL;
                AT91S_PIOB_ASR  = AT91S_LCD_BL;
            }                                              // used actual linear (not by optical effect)
            AT91S_PWM_CR->CH[AT91S_PWM_CH2].CUPD = (PWM_PERIOD_VALUE / 100) * percent;
            break;
    }
}

/***********************************************************
*                  LCD init (EA DOG-M 128-6)               *
***********************************************************/
void GLCD_Init(void)
{
    U08 i;

    GLCDport_InitPWM();
    GLCDport_InitSPI();
    GLCD_Reset();
    GLCDport_Delay(500);
    i = 0; while(i < sizeof(GLCD_InitParams)) GLCDport_Cmd(GLCD_InitParams[i++]);
    GLCDport_Delay(500);
    GLCD_BackLight(50);
    GLCD_ClearScreen();
}

/************************* END ***************************/




/***************************************************************************
****************************************************************************
* TEST MAIN                                                                *
****************************************************************************
***************************************************************************/
// test pattern (only needed parts) of "grafik-memory"
OS_CONST U08 OS_ROMDATA GLCDlogo1p[] = {0x00,0x00,0x80,0x40,0x40,0x20,0x20,0x30,0x10,0x10,0x18,0x08,0x08,0x0C,0x0C,0x0C,0x0C,0x04,0x06,0x86,0xC6,0xC6,0x46,0x42,0xC2,0xC2,0xC3,0xC3,0xE3,0xF3,0xF3,
                                        0xE3,0x63,0x73,0x73,0xF3,0xF6,0xE6,0xC6,0xC6,0x86,0x86,0x0E,0x0E,0x0C,0x0C,0x0C,0x0C,0x1C,0x1C,0x38,0x38,0x30,0x70,0x70,0xE0,0xE0,0xC0,0xC0,0x80,0x80,0x00,0x00,0x00,0x00 };
OS_CONST U08 OS_ROMDATA GLCDlogo2p[] = {0xE0,0xE0,0x70,0xF8,0xE8,0xE8,0xF8,0xB8,0xB8,0x18,0x1C,0x1E,0x0F,0x0D,0x0D,0x0F,0x0F,0x07,0x07,0x07,0x07,0x03,0x03,0x03,0x01,0x01,0x01,0x00,0x00,0x00,0x00,
                                        0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x81,0x83,0xC3,0xC3,0xC7,0xEF,0xEE,0xEE,0xFC,0xB8,0xB8,0x30,0x70,0x60,0xE0,0x00,0x00,0x00,0x01,0x01,0x03,0x0F,0x1F,0xFF,0xFE,0xFC,0xC0 };
OS_CONST U08 OS_ROMDATA GLCDlogo3p[] = {0x03,0x07,0x0C,0x1C,0x18,0x39,0x33,0x63,0xE7,0xC7,0xCF,0x8E,0x1C,0x3C,0x3C,0xF8,0xF8,0xF0,0xF0,0xF0,0xF8,0x38,0x30,0x70,0xF0,0xF8,0xFC,0xFC,0xFC,0xFE,0xF6,
                                        0x06,0x06,0xDE,0xFF,0x7F,0x37,0x37,0x7E,0xFE,0xC0,0xC0,0xE1,0x7F,0x1F,0x07,0x07,0x0F,0x1F,0x38,0x30,0x3F,0x1F,0x80,0xC0,0xC0,0xE0,0xE0,0x70,0x38,0x3E,0x1F,0x0F,0x07,0x00 };
OS_CONST U08 OS_ROMDATA GLCDlogo4p[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x03,0x03,0x07,0x06,0x0F,0x0F,0x0F,0x06,0x07,0x1F,0x38,0x30,0x30,0x1F,0x03,0x03,0x01,0x01,0x03,0x07,
                                        0x06,0x06,0x07,0x13,0x10,0x10,0x10,0x10,0x10,0x10,0x18,0x18,0x08,0x08,0x0C,0x0C,0x0C,0x06,0x06,0x07,0x03,0x03,0x03,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };
OS_CONST U08 OS_ROMDATA GLCDlogo5p[] = {0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x80,0x00,0x80,0x80,0x80,0x80,0x00,0x00,0xE0,0x00,0x80,0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,
                                        0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0xE0,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0xE0,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0x00,
                                        0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0xE0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,
                                        0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00,0xE0,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x00 };
OS_CONST U08 OS_ROMDATA GLCDlogo6p[] = {0x00,0x00,0x1E,0x33,0x70,0x58,0x4C,0x46,0x43,0x21,0x00,0x7F,0x01,0x00,0x00,0x00,0x01,0x7F,0x00,0x00,0x00,0x00,0x7F,0x00,0x7F,0x21,0x40,0x40,0x40,0x40,0x40,0x61,0x3F,0x00,0x1E,0x33,
                                        0x70,0x58,0x4C,0x46,0x43,0x21,0x00,0x1E,0x33,0x40,0x40,0x40,0x40,0x40,0x21,0x7F,0x00,0x1E,0x33,0x40,0x40,0x40,0x40,0x40,0x21,0x7F,0x00,0x1E,0x33,0x70,0x58,0x4C,0x46,0x43,0x21,0x00,
                                        0x1E,0x33,0x40,0x40,0x40,0x40,0x40,0x21,0x7F,0x00,0x00,0x04,0x04,0x04,0x04,0x04,0x00,0x1E,0x21,0x40,0x40,0x40,0x40,0x40,0x21,0x1E,0x00,0x23,0x44,0x4C,0x4C,0x48,0x31,0x00,0x40,0x00,
                                        0x1E,0x33,0x40,0x40,0x40,0x40,0x40,0x21,0x7F,0x00,0x1E,0x33,0x70,0x58,0x4C,0x46,0x43,0x21 };

void main(void)
{
    U08 i;

    GLCD_Init();
    
    GLCD_PutString(" !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrtuvwxyz{|}",0);
    GLCD_UpdateScreen();
    GLCD_ClearScreen();
    GLCD_GotoXY(4,56);
    GLCD_PutString("little world of RTOS", 0);
    GLCD_UpdateScreen();
    GLCD_PutString(" - now we was wrapped!", 0);
    GLCD_UpdateScreen();
    GLCD_Contrast(5);
    GLCD_OnOff(0);
    GLCD_OnOff(1);
    GLCD_SetPixel(64,32);
    GLCD_UpdateScreen();
    GLCD_Contrast(22);
    GLCD_ClearPixel(64,32);
    GLCD_UpdateScreen();
    GLCD_ClearScreen();
    GLCD_GotoXY(4,52);
    GLCD_PutString("little world of RTOS", 0);
    GLCD_UpdateScreen();
    GLCD_PutString(" this line is wrapped!", 0);
    GLCD_UpdateScreen();
    GLCD_ClearScreen();
    for(i = 0; i < sizeof(GLCDlogo1p); i++) GLCDarray[1][33+i] = GLCDlogo1p[i]; GLCDpaged |= (1 << 1);
    for(i = 0; i < sizeof(GLCDlogo2p); i++) GLCDarray[2][33+i] = GLCDlogo2p[i]; GLCDpaged |= (1 << 2);
    for(i = 0; i < sizeof(GLCDlogo3p); i++) GLCDarray[3][33+i] = GLCDlogo3p[i]; GLCDpaged |= (1 << 3);
    for(i = 0; i < sizeof(GLCDlogo4p); i++) GLCDarray[4][33+i] = GLCDlogo4p[i]; GLCDpaged |= (1 << 4);
    for(i = 0; i < sizeof(GLCDlogo5p); i++) GLCDarray[5][i] = GLCDlogo5p[i]; GLCDpaged |= (1 << 5);
    for(i = 0; i < sizeof(GLCDlogo6p); i++) GLCDarray[6][i] = GLCDlogo6p[i]; GLCDpaged |= (1 << 6);
    GLCD_UpdateScreen();
    while(1) { 
        U08 bl; 
        bl= 0;
        while(bl < 100) {
            GLCD_BackLight(bl);
            GLCDport_Delay(100000);
            bl++;
        }
        while(bl > 0) {
            GLCD_BackLight(bl);
            GLCDport_Delay(100000);
            bl--;
        }
    }
}


